<?php
/**
 * Shortcode class
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.0.0
 */
class Page_Generator_Pro_Shortcode {

    /**
     * Holds the class object.
     *
     * @since 1.1.3
     *
     * @var object
     */
    public static $instance;

    /**
     * Holds the base object.
     *
     * @since 1.2.1
     *
     * @var object
     */
    public $base;

    /**
     * Constructor
     *
     * @since   1.0.0
     */
    public function __construct() {

        // Hooks and Filters
        add_action( 'wp_enqueue_scripts', array( $this, 'scripts_css' ) );

        // Register shortcodes
        $this->add_shortcodes();

    }

    /**
     * Registers the shortcodes used by this plugin
     *
     * @since   1.2.0
     *
     * @param   bool    $generating_group   Generating Group
     */
    public function add_shortcodes( $generating_group = false ) {

        add_shortcode( 'page-generator-pro-google-maps', array( $this, 'google_maps' ) );
        add_shortcode( 'page-generator-pro-google-map', array( $this, 'google_maps' ) );
        add_shortcode( 'page-generator-pro-wikipedia', array( $this, 'wikipedia' ) );
        add_shortcode( 'page-generator-pro-yelp', array( $this, 'yelp' ) );
        add_shortcode( 'page-generator-pro-youtube', array( $this, 'youtube' ) );

        // Only register certain shortcodes when we are not generating a group
        if ( ! $generating_group ) {
            add_shortcode( 'page-generator-pro-related-links', array( $this, 'related_links' ) );
        }

    }

    /**
     * Register or enqueue any JS and CSS
     *
     * @since 1.1.0
     */
    public function scripts_css() {

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // If a Google Maps API key has been specified, use it instead of the class default.
        $google_maps_api_key = Page_Generator_Pro_Settings::get_instance()->get_setting( $this->base->plugin->name . '-google', 'google_maps_api_key' );
        $google_maps_api_js_disabled = Page_Generator_Pro_Settings::get_instance()->get_setting( $this->base->plugin->name . '-google', 'google_maps_api_js_disabled' );

        // JS
        wp_enqueue_script( 'jquery' );

        // Only enqueue Google Maps Library if it's not disabled in the Plugin Settings
        if ( ! $google_maps_api_js_disabled ) {
            wp_enqueue_script( $this->base->plugin->name . '-google-maps', 'https://maps.googleapis.com/maps/api/js?v=3.exp&sensor=false' . ( ! empty( $google_maps_api_key ) ? '&key=' . $google_maps_api_key : '' ), false, null, true );    
        }

        wp_enqueue_script( $this->base->plugin->name . '-frontend', $this->base->plugin->url . 'assets/js/min/frontend-min.js', array( 'jquery' ), $this->base->plugin->version, true );

        // CSS
        wp_enqueue_style( $this->base->plugin->name . '-frontend', $this->base->plugin->url . 'assets/css/frontend.css' );

    }

    /**
     * Google Maps Shortcode
	 *
	 * @since  1.0.0
	 *
	 * @param  array   $atts   Shortcode Attributes
	 * @return string 		   Output
     */
    public function google_maps( $atts ) {

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Parse shortcode attributes, defining fallback defaults if required
        $atts = shortcode_atts( array(
            'height'    => 250,
            'term'      => '',
            'location'  => '',
            'zoom'      => 14,
            'maptype'   => 'roadmap',
        ), $atts, $this->base->plugin->name . '-google-map' );

        // Fetch API Key
        // For iframe embeds, there is no usage limit, so we can use the Plugin API Key
        $api_key = Page_Generator_Pro_Geo::get_instance()->google_maps_embed_api_key;

        // Build query
        $q = $atts['location'];
        if ( ! empty( $atts['term'] ) )  {
            $q = $atts['term'] . ' in ' . $q;
        }

        // Build HTML
        $html = '<iframe class="page-generator-pro-map" width="100%" height="' . $atts['height'] . '" frameborder="0" style="border:0" src="https://www.google.com/maps/embed/v1/place?key=' . $api_key . '&q=' . urlencode( $q ) . '&zoom=' . $atts['zoom'] . '&maptype=' . $atts['maptype'] . '" allowfullscreen></iframe>';

        // Return filtered HTML
        return apply_filters( 'page_generator_pro_shortcode_google_maps', $html, $atts );

    }

    /**
     * Related Links Shortcode
     *
     * @since   1.5.9
     *
     * @param   array   $atts   Shortcode Attributes
     * @return  string          Output
     */
    public function related_links( $atts ) {

        global $post;

        // Bail if we're in the WordPress Admin
        if ( is_admin() ) {
            return '';
        }

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Get the Group ID which generated this content
        $group_id = absint( get_post_meta( $post->ID, '_page_generator_pro_group', true ) );

        // Define default shortcode attributes
        $defaults = array(
            // Output and Group ID
            'limit'         => 5,
            'group_id'      => $group_id,

            // Post
            'post_type'     => $post->post_type,
            'post_status'   => 'publish',
            'post__not_in'  => array( $post->ID ),

            // Order
            'orderby'       => 'rand',
            'order'         => 'ASC',
            
            // Author
            'author'        => '',

            // Taxonomy
            // @TODO

        );

        // Allow default settings to be filtered
        $defaults = apply_filters( 'page_generator_pro_shortcode_related_links_defaults', $defaults, $atts, $post );

        // Parse shortcode attributes, using defaults if required
        $atts = shortcode_atts( $defaults, $atts, $this->base->plugin->name . '-related-links' );

        // If certain attributes are empty, but require a value, set them now
        if ( empty( $atts['group_id'] ) ) {
            $atts['group_id'] = $group_id;
        }

        // Fetch other Pages / Posts generated in this Group
        $posts = new WP_Query( array(
            'post_type'     => $atts['post_type'],
            'post_status'   => $atts['post_status'],
            'post__not_in'  => $atts['post__not_in'],
            'posts_per_page'=> absint( $atts['limit'] ),
            'orderby'       => $atts['orderby'],
            'order'         => $atts['order'],
            'meta_query'    => array(
                array(
                    'key'   => '_page_generator_pro_group',
                    'value' => absint( $atts['group_id'] ),
                ),
            ),

            // For performance, just return the Post ID and don't update meta or term caches
            'fields'                => 'ids',
            'cache_results'         => false,
            'update_post_meta_cache'=> false,
            'update_post_term_cache'=> false,
        ) );

        // If no other Pages / Posts found, bail
        if ( count( $posts->posts ) == 0 ) {
            return '';
        }

        // Build link list
        $html = '<ul class="page-generator-pro-related-links">';
        foreach ( $posts->posts as $post_id ) {
            $html .= '<li><a href="' . get_permalink( $post_id ) . '">' . get_the_title( $post_id ) . '</a></li>';
        }
        $html .= '</ul>';

        // Return filtered HTML
        return apply_filters( 'page_generator_pro_shortcode_related_links', $html, $atts, $posts, $post );

    }

    /**
     * Wikipedia Shortcode
	 *
	 * @since  1.0.0
	 *
     * @param  array   $atts   Shortcode Attributes
     * @return string          Output
     */
    public function wikipedia( $atts ) {

        // Check DOMDocument installed
        if ( ! class_exists( 'DOMDocument' ) ) {
            return '';
        }

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Parse shortcode attributes, defining fallback defaults if required
        $atts = shortcode_atts( array(
            'term'      => '',
            'language'  => 'en',
            'sections'  => 2, // Number of wikipedia sections to include
        ), $atts, $this->base->plugin->name . '-wikipedia' );

    	// Check if an article exists on Wikipedia for the given term
        $response = wp_remote_get( 'http://' . $atts['language'] . '.wikipedia.org/w/api.php?section=0&action=parse&page=' . $atts['term'] . '&format=json&prop=text&redirects' );

        // Bail if an error occured
        if ( is_wp_error( $response ) ) {
            return '';
        }
        if ( $response['response']['code'] != 200 ) {
			return '';
		}

		// Check status
        $body = wp_remote_retrieve_body( $response );
        $check = json_decode( $body );
        if ( isset( $check->error ) ) {
        	return '';
        }
        
        // Get data from Wikipedia
        $url = 'http://' . $atts['language'] . '.wikipedia.org/wiki/' . str_replace(' ', '_', $atts['term'] );
        $response = wp_remote_get( $url );

        // Bail if an error occured
        if ( is_wp_error( $response ) ) {
            return '';
        }
        if ( $response['response']['code'] != 200 ) {
            return '';
        }

        // Get body
        $body = wp_remote_retrieve_body( $response );
        if ( empty( $body ) ) {
            return '';
        }

        // Parse into DOMDocument
		$dom = new DOMDocument();
        $dom->preserveWhiteSpace = false; 

        // Convert encoding to UTF-8 if php-mbstring is installed
        if ( function_exists( 'mb_convert_encoding' ) ) {
            @$dom->loadHTML( mb_convert_encoding( $body, 'HTML-ENTITIES', 'UTF-8' ) );
        } else {
            // Cannot guarantee this works as mb_convert_encoding is not available
            @$dom->loadHTML( $body );
        }

        // Extract paragraphs that contain the main content
        $paragraphs = $dom->getElementById( 'mw-content-text' )->getElementsByTagName( 'p' );
        
        // Build paragraphs
        $x = 1;
        $build = '';
        foreach( $paragraphs as $paragraph ) {
            // Skip if the paragraph is empty
            if ( empty( $paragraph->nodeValue ) ) {
                continue;
            }

            // Skip if this entire paragraph's nodeValue matches the keyword
            // e.g. the city name
            if ( strpos( $atts['term'], trim( $paragraph->nodeValue ) ) !== false ) {
                continue;
            }

            // Skip if the paragraph starts with certain characters
            if ( strpos( trim( $paragraph->nodeValue ), '[[' ) !== false ) {
                continue;
            }

           	if ( $x <= $atts['sections'] ) {
				$build .= "<p>" . $paragraph->nodeValue . "</p>";
                $x++;
			} else {
				break;
            }
            
		}

		// Check WikiPedia returned content in our build process
		if ( empty( $build ) ) {
			return '';
		}

        // Build HTML
        $html = '<div class="' . $this->base->plugin->name . '-wikipedia">' . $build . '</div>';

        // Return filtered HTML
        return apply_filters( 'page_generator_pro_shortcode_wikipedia', $html, $atts, $build );

    }

    /**
     * Yelp Shortcode
	 *
	 * @since 1.0.0
	 *
	 * @param  array   $atts   Shortcode Attributes
	 * @return string 		   Output
     */
    public function yelp( $atts ) {

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Get instances
        $settings_instance = Page_Generator_Pro_Settings::get_instance();

        // Parse shortcode attributes, defining fallback defaults if required
        $atts = shortcode_atts( array(
            'term'      => '',
            'location'  => '',
            'limit'     => 5,
        ), $atts, $this->base->plugin->name . '-yelp' );

    	// Send request to Yelp API
        $yelp = Page_Generator_Pro_Yelp::get_instance();
        $results = $yelp->businesses_search( $atts['term'], $atts['location'], $atts['limit'] );
        
        // Check for errors from the Yelp API
        if ( is_wp_error( $results ) ) {
        	if ( is_user_logged_in() ) {
        		// Display error
        		return 'Yelp API: ' . $results->get_error_message();
        	}

        	// Just return a blank string
        	return '';
        }

        // Check if any businesses were found
		if ( ! isset( $results->businesses ) || count( $results->businesses ) == 0 ) {
			return '';
		}

        // Iterate through results, building HTML
        $html = '<div class="' . $this->base->plugin->name . '-yelp">';
		foreach ( $results->businesses as $count => $business ) {

			$html .= '<div class="business">
				<div class="name">' . $business->name . '</div>
				<div class="rating">
                    <img src="' .$this->base->plugin->url . '/assets/images/yelp-ratings/' . $business->rating . '.png" />' 
                    . $business->review_count . ' ' . __( 'reviews', $this->base->plugin->name ) . '
                </div>
				<div class="categories">';
			
			// Categories
			$total_categories = count( $business->categories );
			foreach ( $business->categories as $cCount => $category ) {
                $html .= $category->title;
				if ( ( $cCount + 1 ) != $total_categories ) {
					$html .= ', ';
				}
			}
				
			$html .= '
				</div>
				<div class="phone">' . $business->phone . '</div>
				
				<div class="address">';
			
			// Address
			$total_address_lines = count( $business->location->display_address );
			foreach ( $business->location->display_address as $aCount => $address ) {
				$html .= $address;
				if ( ( $aCount + 1 ) != $total_address_lines ) {
					$html .= ', ';
				}
			}
			
			$html .= '
				</div>
			</div>';
			
			// Check if limit reached
			if ( ( $count + 1 ) == $atts['limit'] ) {
				break;
			}
		}
		
        // Filter HTML
        $html = apply_filters( 'page_generator_pro_shortcode_yelp', $html, $atts );

        // Add Yelp logo. This is required to meet the display requirements below, which is why this is done after filtering
        // http://www.yelp.co.uk/developers/getting_started/display_requirements
        $html .= '<a href="https://www.yelp.com" target="_blank"><img src="https://s3-media1.ak.yelpcdn.com/assets/2/www/img/55e2efe681ed/developers/yelp_logo_50x25.png" /></a>
        </div>';

        // Return
        return $html;

    }

    /**
     * YouTube Shortcode
     *
     * @since   1.2.0
     *
     * @param   array   $atts   Shortcode Attributes
     * @return  string          Output
     */
    public function youtube( $atts ) {

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Try to get the post ID
        // If we're generating a new page, this might not be possible
        global $post;
        if ( ! empty( $post ) ) {
            $post_id = $post->ID;
        } else {
            $post_id = 0;
        }

        // Parse shortcode attributes, defining fallback defaults if required
        $atts = shortcode_atts( array(
            'term'          => '',
            'location'      => 0,
        ), $atts, $this->base->plugin->name . '-youtube' );

        // If our term is a location, get its latitude and longitude now
        if ( $atts['location'] ) {
            // Setup instance
            $geo_instance = Page_Generator_Pro_Geo::get_instance();

            // If a Google Geocoding API key has been specified, use it instead of the class default.
            $google_geocoding_api_key = Page_Generator_Pro_Settings::get_instance()->get_setting( $this->base->plugin->name . '-google', 'google_geocoding_api_key' );
            if ( ! empty( $google_geocoding_api_key ) ) {
                $geo_instance->api_key = $google_geocoding_api_key;
            }

            $lat_lng = $geo_instance->google_get_lat_lng( $atts['location'] );
        } else {
            $lat_lng = false;
        }

        // Setup YouTube instance
        $youtube = Page_Generator_Pro_Youtube::get_instance();
        
        // If a YouTube Data API key has been specified, use it instead of the class default.
        $youtube_data_api_key = Page_Generator_Pro_Settings::get_instance()->get_setting( $this->base->plugin->name . '-google', 'youtube_data_api_key' );
        if ( ! empty( $youtube_data_api_key ) ) {
            $youtube->api_key = $youtube_data_api_key;
        }

        // Run query
        $videos = $youtube->search( $atts['term'], $lat_lng );
        if ( is_wp_error( $videos ) || ! is_array( $videos ) || count( $videos ) == 0 ) {
            // Couldn't fetch videos from YouTube, so don't show a video at all.
            return '';
        }

        // Pick a video at random from the resultset
        if ( count( $videos ) == 1 ) {
            $video_index = 0;
        } else {
            $video_index = rand( 0, ( count( $videos ) - 1 ) );
        }

        // Get video URL
        $url = $videos[ $video_index ]['url'];

        // Return filtered URL
        return apply_filters( 'page_generator_pro_shortcode_youtube', $url, $atts, $videos, $video_index );

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.3
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}

// Load the class
$page_generator_pro_shortcode = Page_Generator_Pro_Shortcode::get_instance();