<?php
/**
 * Page Builders class
 * 
 * @package  Page Generator Pro
 * @author   Tim Carr
 * @version  1.3.7
 */
class Page_Generator_Pro_PageBuilders {

    /**
     * Holds the class object.
     *
     * @since   1.3.7
     *
     * @var     object
     */
    public static $instance;

    /**
     * Constructor
     *
     * @since   1.3.7
     */
    public function __construct() {

        // Register Support for Page Builders
        add_filter( 'avf_builder_boxes', array( $this, 'register_avia_layout_builder_support' ), 10, 1 );
        add_filter( 'fl_builder_post_types', array( $this, 'register_beaver_builder_support' ) );
        add_action( 'add_meta_boxes', array( $this, 'register_betheme_support' ) );
        add_filter( 'et_builder_post_types', array( $this, 'register_divi_support' ) );
        add_action( 'page_generator_pro_groups_add_meta_boxes', array( $this, 'register_divi_metabox_support' ) );
        add_filter( 'et_pb_show_all_layouts_built_for_post_type', array( $this, 'register_divi_layout_support' ) );
        add_filter( 'fusion_builder_allowed_post_types', array( $this, 'register_fusion_builder_support' ) );
        add_filter( 'fusion_builder_default_post_types', array( $this, 'register_fusion_builder_support' ) );
        add_action( 'init', array( $this, 'register_visual_composer_support' ) );
        add_filter( 'dslc_can_edit_in_lc', array( $this, 'register_live_composer_page_builder_support' ), 10, 2 );

        // Stop Themes and other Plugins disabling the frontend editor
        add_action( 'vc_before_init', array( $this, 'visual_composer_enable_frontend' ), PHP_INT_MAX );
        add_action( 'vc_after_init', array( $this, 'visual_composer_enable_frontend' ), PHP_INT_MAX );
        
    }

    /**
     * Allows the Avia Layout Builder (which coems with the Enfold Theme) to inject
     * its Page Builder into Page Generator Pro's Groups
     *
     * @since   1.5.6
     *
     * @param   array   $meta_boxes     Meta Boxes
     * @return  array                   Meta Boxes
     */
    public function register_avia_layout_builder_support( $meta_boxes ) {

        // Bail if no Meta Boxes exist
        if ( empty( $meta_boxes ) ) {
            return $meta_boxes;
        }

        // Define the Avia Meta Box IDs
        $avia_meta_box_ids = apply_filters( 'page_generator_pro_pagebuilders_register_avia_layout_builder_support', array(
            'avia_builder',
            'layout',
            'preview',
            'hierarchy',
        ), $meta_boxes );

        // Iterate through the existing Meta Boxes, to find the Avia specific ones
        foreach ( $meta_boxes as $key => $meta_box ) {
            // Skip if the ID isn't one we are looking for
            if ( ! in_array( $meta_box['id'], $avia_meta_box_ids ) ) {
                continue;
            }

            // Add Page Generator Pro's Groups to the 'page' array
            $meta_boxes[ $key ]['page'][] = 'page-generator-pro';
        }

        return $meta_boxes;

    }

    /**
     * Allows Beaver Builder to inject its Page Builder
     * into Page Generator Pro's Groups
     *
     * @since   1.3.7
     *
     * @param   array   $post_types     Post Types Supporting Beaver Builder
     * @return  array                   Post Types Supporting Beaver Builder
     */
    public function register_beaver_builder_support( $post_types ) {

        $post_types[] = 'page-generator-pro';
        return $post_types;

    }

    /**
     * Allows BeTheme's Muffin Builder Meta Box to be output on Page Generator Pro's Groups
     *
     * @since   1.3.1
     */
    public function register_betheme_support() {

        /**
         * BeTheme doesn't provide any actions or filters; but does populate the global
         * variable on the admin_menu action, which comprises of the necessary arguments
         * for registering the Page Builder on Pages.
         *
         * We check for its existence, copy the var, tweak the arguments and then call
         * add_meta_box to register it against our plugin
         */
        global $mfn_page_meta_box;

        // Bail if meta box array doesn't exist
        if ( ! isset( $mfn_page_meta_box ) ) {
            return;
        }
        if ( empty( $mfn_page_meta_box ) ) {
            return;
        }
        if ( ! is_array( $mfn_page_meta_box ) ) {
            return;
        }

        // Copy
        $mfn_groups_meta_box = $mfn_page_meta_box;

        // Change some settings
        $mfn_groups_meta_box['id'] = 'mfn-meta-page-generator-pro';
        $mfn_groups_meta_box['page'] = 'page-generator-pro';

        // Add meta box
        add_meta_box(
            $mfn_groups_meta_box['id'], 
            $mfn_groups_meta_box['title'], 
            'mfn_page_show_box', 
            $mfn_groups_meta_box['page'], 
            $mfn_groups_meta_box['context'], 
            $mfn_groups_meta_box['priority']
        );

    }

    /**
     * Allows The Divi Builder (and therefore Divi Theme 3.0+) to inject its Page Builder
     * into Page Generator Pro's Groups
     *
     * @since   1.2.7
     *
     * @param   array   $post_types     Post Types Supporting Divi
     * @return  array                   Post Types Supporting Divi
     */
    public function register_divi_support( $post_types ) {

        $post_types[] = 'page-generator-pro';
        return $post_types;

    }

    /**
     * Allows The Divi Builder (and therefore Divi Theme 3.0+) to inject its Page Builder
     * Meta Box into this Plugin's enabled Custom Post Types
     *
     * @since   1.4.1
     *
     * @param   obj     $post_type_instance     Post Type Instance
     */
    public function register_divi_metabox_support( $post_type_instance ) {

        // Don't need to do anything if we're not in the admin interface
        if ( ! is_admin() ) {
            return;
        }

        // Don't add the meta box if Divi Builder isn't active
        if ( ! function_exists( 'et_single_settings_meta_box' ) ) {
            return;
        }

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Add Meta Box
        // We don't use add_meta_box( 'et_settings_meta_box'... because we need to change
        // the Post Type = post, so that all settings display, without changing the global $post
        add_meta_box( 
            'et_settings_meta_box',
            __( 'Divi Settings', $this->base->plugin->name ), 
            array( $this, 'output_divi_metabox' ), 
            $post_type_instance->post_type_name,
            'side',
            'high'
        );

    }

    /**
     * Outputs the Divi Settings Metabox
     *
     * @since   1.6.4
     */
    public function output_divi_metabox() {

        // Trick Divi into outputting Post settings
        global $post;
        $new_post = $post;
        $new_post->post_type = 'post';

        // Call metabox function directly.
        et_single_settings_meta_box( $new_post );        

    }

    /**
     * Allows The Divi Builder (and therefore Divi Theme 3.0+) to inject its Page Builder Layouts
     * into the chosen Custom Post Types
     *
     * @since   1.4.1
     *
     * @param   mixed   $post_types     string | array
     * @return  array                   Post Types to get Layouts from
     */
    public function register_divi_layout_support( $post_types ) {

        // Bail if we're not on Page Generator Pro
        if ( $post_types != 'page-generator-pro' ) {
            return $post_types;
        }

        // If $post_types isn't an array, make it one
        if ( ! is_array( $post_types ) ) {
            $post_types = array( $post_types );
        }

        // Fetch Public Post Types
        $public_post_types = Page_Generator_Pro_Common::get_instance()->get_post_types();

        // Add all Public Post Types to $post_types
        foreach ( $public_post_types as $public_post_type ) {
            // Add Custom Post Type to Divi, so the Page Builder displays
            $post_types[] = $public_post_type->name;
        }

        // Remove duplicates
        $post_types = array_unique( $post_types );

        // Return
        return $post_types;

    }

    /**
     * Allows Fusion Builder (and therefore Avada Theme) to inject its Page Builder
     * into Page Generator Pro's Groups
     *
     * @since   1.2.8
     *
     * @param   array   $post_types     Post Types Supporting Divi
     * @return  array                   Post Types Supporting Divi
     */
    public function register_fusion_builder_support( $post_types ) {

        $post_types[] = 'page-generator-pro';
        return $post_types;

    }

    /**
     * Allows Visual Composer to inject its Page Builder into Page Generator Pro's Groups,
     * by adding the vc_access_rules_post_types/page-generator-pro Role for Administrators
     * if we're in the WP Admin.
     *
     * @since   1.3.7
     */
    public function register_visual_composer_support() {

        // Bail if not in the WordPress Admin
        if ( ! is_admin() ) {
            return;
        }

        // Fetch the roles that need to be granted Visual Composer access.
        // Devs / addons can override this
        $roles = apply_filters( 'page_generator_pro_pagebuilders_register_visual_composer_support_roles', array(
            'administrator',
            'editor',
        ) );

        foreach ( (array) $roles as $role_name ) {
            // Skip if this role already has the capabilities
            $role = get_role( $role_name );
            if ( isset( $role->capabilities['vc_access_rules_post_types/page-generator-pro'] ) &&
                 isset( $role->capabilities['vc_access_rules_frontend_editor'] ) ) {
                continue;
            }   

            // Add the capabilities to this role
            // Both are required to ensure correct working functionality!
            $role->add_cap( 'vc_access_rules_post_types/page-generator-pro' );
            $role->add_cap( 'vc_access_rules_frontend_editor' );
               
        }

    }

    /**
     * Stop Themes and other Plugins disabling Visual Composer on all other Post Types except their own.
     *
     * Ensures that the 'Edit with Visual Composer' is always available on Groups
     *
     * @since   1.4.5
     */
    public function visual_composer_enable_frontend() {

        vc_disable_frontend( false );

    }

    /**
     * Allows Live Composer to inject its Page Builder into Page Generator Pro's Groups,
     *
     * @since   1.6.8
     */
    public function register_live_composer_page_builder_support( $can_edit, $post_type ) {

        // Bail if we're not on a Page Generator Pro group
        if ( $post_type != 'page-generator-pro' ) {
            return $can_edit;
        }

        // Enable Live Composer on Page Generator Pro Groups
        return true;

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.3.7
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}

// Load the class
$page_generator_pro_pagebuilders   = Page_Generator_Pro_PageBuilders::get_instance();