<?php
/**
 * Yelp API v3 Wrapper
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.0.0
 */
class Page_Generator_Pro_Yelp {

    /**
     * Holds the class object.
     *
     * @since   1.4.5
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the base class object.
     *
     * @since   1.4.5
     *
     * @var     object
     */
    public $base;

    /**
     * API Endpoint
     *
     * @since   1.4.5
     *
     * @var     string
     */
    public $endpoint = 'https://api.yelp.com/v3/';

    /**
     * Performs a GET request to /businesses/search
     *
     * @since   1.4.5
     *
     * @param   string  $term       Search Terms
     * @param   string  $location   Location
     * @param   int     $limit      Number of Results
     * @return  mixed               WP_Error | array
     */
    public function businesses_search( $term, $location, $limit = 5 ) {

        return $this->get( 'businesses/search', array(
            'term'      => $term,
            'location'  => $location,
            'limit'     => $limit,
        ) );

    }

    /**
     * Checks settings for an Access Token, falling back to calling
     * the WP Zinc oAuth Gateway to fetch a new Access Token if
     * required.
     *
     * @since   1.4.5
     */
    private function get_access_token() {

        // Get base instance
        $this->base = Page_Generator_Pro::get_instance();

        // Get settings
        $access_token = Page_Generator_Pro_Settings::get_instance()->get_settings( $this->base->plugin->name . '-yelp' );

        // If settings are empty or an array, fetch a v3 token
        if ( empty( $access_token ) || is_array( $access_token ) ) {
            $response = wp_remote_post( 'https://www.wpzinc.com/?oauth=yelp' );

            // If an error, return it
            if ( is_wp_error( $response ) ) {
                return $response;
            }

            // Get response code
            $response_code = wp_remote_retrieve_response_code( $response );
            $response_message = wp_remote_retrieve_response_message( $response );

            // Bail if response code isn't 200 OK
            if ( $response_code != 200 ) {
                return new WP_Error( 'page_generator_pro_yelp_get_access_token', $response_code . ' ' . $response_message );
            }

            // Get body
            $data = wp_remote_retrieve_body( $response );

            // Bail if not successful
            $data = json_decode( $data );
            if ( ! $data->success ) {
                return new WP_Error( 'page_generator_pro_yelp_get_access_token', $data->data );
            }

            // Store access token
            Page_Generator_Pro_Settings::get_instance()->update_settings( $this->base->plugin->name . '-yelp', $data->data->access_token );
        }

        // Return access token from settings
        return Page_Generator_Pro_Settings::get_instance()->get_settings( $this->base->plugin->name . '-yelp' );

    }

    /**
     * Performs a GET request
     *
     * @since   1.4.5
     *
     * @param   string  $endpoint   Endpoint
     * @param   array   $arguments  Arguments
     * @return  mixed               WP_Error | array
     */
    private function get( $endpoint, $arguments ) {

        // Get Access Token
        $access_token = $this->get_access_token();

        // If an error was returned, bail
        if ( is_wp_error( $access_token ) ) {
            return $access_token;
        }

        // Perform GET Request
        $response = wp_remote_get( $this->endpoint . $endpoint, array(
            'headers'   => array(
                'Authorization' => 'Bearer ' . $access_token,

            ),
            'body'      => $arguments,
            'user-agent'=> 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_9_2) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/34.0.1847.131 Safari/537.36',
        ) );

        // Bail if an error
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        // Get response code
        $response_code = wp_remote_retrieve_response_code( $response );
        $response_message = wp_remote_retrieve_response_message( $response );

        // Bail if response code isn't 200 OK
        if ( $response_code != 200 ) {
            return new WP_Error( 'page_generator_pro_yelp_get_error', $response_code . ' ' . $response_message );
        }

        // Get body
        $data = wp_remote_retrieve_body( $response );

        // Return data
        return json_decode( $data );

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.2.0
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}