<?php
/**
 * Webanalyzer API class
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.7.1
 */
class Page_Generator_Pro_Webanalyzer_API {

    /**
     * Holds the API endpoint
     *
     * @since   1.7.1
     *
     * @var     string
     */
    private $api_endpoint = 'https://www.wpzinc.com/?webanalyzer_api=1';

    /**
     * Returns synonyms for the given keyword
     *
     * @since   1.7.1
     *
     * @param   string  $keyword    Keyword
     * @return  mixed               WP_Error | array
     */
	public function synonyms( $keyword ) {

        return $this->post( 'synonyms', array(
            'keyword' => $keyword,
        ) );

	}

	/**
	 * Private function to perform a GET request
	 *
	 * @since  1.7.1
	 *
	 * @param  string  $cmd 	   Command (required)
	 * @param  array   $params 	   Params (optional)
	 * @return mixed 		       WP_Error | object
	 */
	private function get( $cmd, $params = array() ) {

		return $this->request( $cmd, 'get', $params );

	}

	/**
	 * Private function to perform a POST request
	 *
     * @since  1.7.1
     *
     * @param  string  $cmd        Command (required)
     * @param  array   $params     Params (optional)
     * @return mixed               WP_Error | object
     */
	private function post( $cmd, $params = array() ) {

		return $this->request( $cmd, 'post', $params );

	}

	/**
     * Main function which handles sending requests to the API
     *
     * @since   1.7.1
     *
     * @param   string  $cmd        Command
     * @param   string  $method 	Method (get|post)
     * @param   array   $params 	Parameters (optional)
     * @return mixed                WP_Error | object
     */
    private function request( $cmd, $method = 'get', $params = array() ) {

        // Define timeout, in seconds
        $timeout = apply_filters( 'page_generator_pro_webanalyzer_api_request_timeout', 10 );

        // Request via WordPress functions
        $result = $this->request_wordpress( $this->api_endpoint, $cmd, $method, $params, $timeout );

        // Request via cURL if WordPress functions failed
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
            if ( is_wp_error( $result ) ) {
                $result = $this->request_curl( $url, $method, $params, $timeout );
            }
        }

        // Result will be WP_Error or the data we expect
        return $result;

    }

    /**
     * Performs POST and GET requests through WordPress wp_remote_post() and
     * wp_remote_get() functions
     *
     * @since   1.7.1
     *
     * @param   string  $url        URL
     * @param   string  $cmd        API Command
     * @param   string  $method     Method (post|get)
     * @param   array   $params     Parameters
     * @param   int     $timeout    Timeout, in seconds (default: 10)
     * @return  mixed               WP_Error | object
     */
    private function request_wordpress( $url, $cmd, $method, $params, $timeout = 10 ) {

        // Send request
        switch ( $method ) {
            /**
             * GET
             */
            case 'get':
                $result = wp_remote_get( $url, array(
                    'body'      => array(
                        'endpoint'  => $cmd,
                        'params'    => $params,
                    ),
                    'timeout'   => $timeout,
                ) );
                break;
            
            /**
             * POST
             */
            case 'post':
                $result = wp_remote_post( $url, array(
                    'body'      => array(
                        'endpoint'  => $cmd,
                        'params'    => $params,
                    ),
                    'timeout'   => $timeout,
                ) );
                break;
        }

        // If an error occured, return it now
        if ( is_wp_error( $result ) ) {
            return $result;
        }

        // If the HTTP code isn't 200, something went wrong
        if ( $result['response']['code'] != 200 ) {
            // Decode error message
            $body = json_decode( $result['body'] );

            // Define the error message
            $message = array();
            if ( isset( $body->error ) ) {
                $message[] = $body->error;
            }
            if ( isset( $body->message ) ) {
                $message[] = $body->message;
            }

            // Return WP_Error
            return new WP_Error( 
                $result['response']['code'], 
                'Webanalyzer API Error: HTTP Code ' . $result['response']['code'] . '. #' . $body->code . ' - ' . implode( "\n", $message ) 
            );
        }

        // All OK, return response
        return json_decode( $result['body'] );

    }

    /**
     * Performs POST and GET requests through PHP's curl_exec() function.
     *
     * If this function is called, request_wordpress() failed, most likely
     * due to a DNS lookup failure or CloudFlare failing to respond.
     *
     * @since   1.7.1
     *
     * @param   string  $url        URL
     * @param   string  $method     Method (post|get)
     * @param   array   $params     Parameters
     * @param   int     $timeout    Timeout, in seconds (default: 10)
     * @return  mixed               WP_Error | object
     */
    private function request_curl( $url, $cmd, $method, $params, $timeout = 10 ) {

        // Init
        $ch = curl_init();

        // Set request specific options
        switch ( $method ) {
            /**
             * GET
             */
            case 'get':
                curl_setopt_array( $ch, array(
                    CURLOPT_URL             => $url . '&' . http_build_query( array(
                        'endpoint'  => $cmd,
                        'params'    => $params,
                    ) ),
                ) );
                break;

            /**
             * POST
             */
            case 'post':
                curl_setopt_array( $ch, array(
                    CURLOPT_URL             => $url,
                    CURLOPT_POST            => true,
                    CURLOPT_POSTFIELDS      => http_build_query( array(
                        'endpoint'  => $cmd,
                        'params'    => $params,
                    ) ),
                ) );
                break;
        }

        // Set shared options
        curl_setopt_array( $ch, array(
            CURLOPT_RETURNTRANSFER  => true,
            CURLOPT_HEADER          => false,
            CURLOPT_FOLLOWLOCATION  => true,
            CURLOPT_MAXREDIRS       => 10,
            CURLOPT_CONNECTTIMEOUT  => $timeout,
            CURLOPT_TIMEOUT         => $timeout,
        ) );

        // Execute
        $result     = curl_exec( $ch );
        $http_code  = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
        $error      = curl_error( $ch );
        curl_close( $ch );

        // If our error string isn't empty, something went wrong
        if ( ! empty( $error ) ) {
            return new WP_Error( 'page_generator_pro_webanalyzer_api_request_curl', $error );
        }

        // If HTTP code isn't 200, something went wrong
        if ( $http_code != 200 ) {
            // Decode error message
            $result = json_decode( $result );

            // Return basic WP_Error if we don't have any more information
            if ( is_null( $result ) ) {
                return new WP_Error(
                    $http_code,
                    'Webanalyzer API Error: HTTP Code ' . $http_code . '. Sorry, we don\'t have any more information about this error. Please try again.'
                );
            }

            // Define the error message
            $message = array();
            if ( isset( $result->error ) ) {
                $message[] = $result->error;
            }
            if ( isset( $result->message ) ) {
                $message[] = $result->message;
            }

            // Return WP_Error
            return new WP_Error( $http_code, 'Webanalyzer API Error: HTTP Code ' . $http_code . '. #' . $result->code. ' - ' . implode( "\n", $message )  );
        }
        
        // All OK, return response
        return json_decode( $result );

    }

}