<?php
/**
 * Webanalyzer API class
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.7.1
 */
class Page_Generator_Pro_Webanalyzer_API {

    /**
     * Holds the class object.
     *
     * @since   1.7.1
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the API endpoint
     *
     * @since   3.4.7
     *
     * @var     string
     */
    private $api_endpoint = 'https://thesaurus.com/';

    /**
     * Returns synonyms for the given keyword
     *
     * @since   1.7.1
     *
     * @param   string  $keyword    Keyword
     * @return  mixed               WP_Error | array
     */
	public function synonyms( $keyword ) {

        return $this->get( 'browse/' . $keyword );

	}

	/**
     * Returns a list of Social Media Profiles attached to the Buffer Account.
     *
     * @since   3.0.0
     *
     * @param   bool    $force                      Force API call (false = use WordPress transient)
     * @param   int     $transient_expiration_time  Transient Expiration Time
     * @return  mixed                               WP_Error | Profiles object
     */
	public function profiles( $force = false, $transient_expiration_time ) {

        // Check access token
        if ( ! $this->check_access_token_exists() ) {
            return false;
        }

        // Setup profiles array
        $profiles = array();

        // Check if our WordPress transient already has this data.
        // This reduces the number of times we query the API
        if ( $force || false === ( $profiles = get_transient( $this->base->plugin->name . '_buffer_api_profiles' ) ) ) {
            // Get profiles
            $results = $this->get( 'profiles.json?subprofiles=1' );

            // Check for errors
            if ( is_wp_error( $results ) ) {
                return $results;
            }

            // Check data is valid
            foreach ( $results as $result ) {
                // We don't support Instagram or Pinterest in the Free version, as there's no Featured Image option.
                if ( class_exists( 'WP_To_Buffer' ) ) {
                    if ( $result->service == 'instagram' || $result->service == 'pinterest' ) {
                        continue;
                    }
                }
                
                // Add profile to array
                $profiles[ $result->id ] = array(
                    'id'                => $result->id,
                    'formatted_service' => $result->formatted_service,
                    'formatted_username'=> $result->formatted_username,
                    'avatar'            => $result->avatar,
                    'service'           => $result->service,
                );

                // Pinterest: add subprofiles
                if ( isset( $result->subprofiles ) && count( $result->subprofiles ) > 0 ) {
                    $profiles[ $result->id ]['subprofiles'] = array();
                    foreach ( $result->subprofiles as $sub_profile ) {
                        $profiles[ $result->id ]['subprofiles'][ $sub_profile->id ] = array(
                            'id'        => $sub_profile->id,
                            'name'      => $sub_profile->name,
                            'service'   => $sub_profile->service,
                        );
                    }
                }
            }
            
            // Store profiles in transient
            set_transient( $this->base->plugin->name . '_buffer_api_profiles', $profiles, $transient_expiration_time );
        }

		// Return results
        return $profiles;

	}

    /**
     * Creates an Update
     *
     * @since   3.0.0
     *
     * @return  mixed   WP_Error | Update object
     */
    public function updates_create( $params ) {

        // Check access token
        if ( ! $this->check_access_token_exists() ) {
            return false;
        }

        // Send request
        $result = $this->post( 'updates/create.json', $params );

        // Bail if the result is an error
        if ( is_wp_error( $result ) ) {
            return $result;
        }

        // Return array of just the data we need to send to the Plugin
        return array(
            'profile_id'        => $result->updates[0]->profile_id,
            'message'           => $result->message,
            'status_text'       => $result->updates[0]->text,
            'status_created_at' => $result->updates[0]->created_at,
            'due_at'            => $result->updates[0]->due_at,
        );

    }

	/**
	 * Private function to perform a GET request
	 *
	 * @since  3.0.0
	 *
	 * @param  string  $cmd 	   Command (required)
	 * @param  array   $params 	   Params (optional)
	 * @return mixed 		       WP_Error | object
	 */
	private function get( $cmd, $params = array() ) {

		return $this->request( $cmd, 'get', $params );

	}

	/**
	 * Private function to perform a POST request
	 *
     * @since  3.0.0
     *
     * @param  string  $cmd        Command (required)
     * @param  array   $params     Params (optional)
     * @return mixed               WP_Error | object
     */
	private function post( $cmd, $params = array() ) {

		return $this->request( $cmd, 'post', $params );

	}

	/**
     * Main function which handles sending requests to the Buffer API
     *
     * @since   3.0.0
     *
     * @param   string  $cmd        Command
     * @param   string  $method 	Method (get|post)
     * @param   array   $params 	Parameters (optional)
     * @return mixed                WP_Error | object
     */
    private function request( $cmd, $method = 'get', $params = array() ) {

    	// Check required parameters exist
    	if ( empty( $this->access_token ) ) {
            return new WP_Error( 'missing_access_token', __( 'No access token was specified' ) );
    	}

    	// Add access token to command, depending on the command's format
    	if ( strpos ( $cmd, '?' ) !== false ) {
    		$cmd .= '&access_token=' . $this->access_token;
    	} else {
    		$cmd .= '?access_token=' . $this->access_token;
    	}

    	// Build endpoint URL
    	$url = $this->api_endpoint . $this->api_version . '/' . $cmd;

        // Define timeout, in seconds
        $timeout = apply_filters( $this->base->plugin->name . '_buffer_api_request', 10 );

        // Request via WordPress functions
        $result = $this->request_wordpress( $url, $method, $params, $timeout );

        // Request via cURL if WordPress functions failed
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG === true ) {
            if ( is_wp_error( $result ) ) {
                $result = $this->request_curl( $url, $method, $params, $timeout );
            }
        }

        // Result will be WP_Error or the data we expect
        return $result;

    }

    /**
     * Performs POST and GET requests through WordPress wp_remote_post() and
     * wp_remote_get() functions
     *
     * @since   3.2.6
     *
     * @param   string  $url        URL
     * @param   string  $method     Method (post|get)
     * @param   array   $params     Parameters
     * @param   int     $timeout    Timeout, in seconds (default: 10)
     * @return  mixed               WP_Error | object
     */
    private function request_wordpress( $url, $method, $params, $timeout = 10 ) {

        // Send request
        switch ( $method ) {
            /**
             * GET
             */
            case 'get':
                $result = wp_remote_get( $url, array(
                    'body'      => $params,
                    'timeout'   => $timeout,
                ) );
                break;
            
            /**
             * POST
             */
            case 'post':
                $result = wp_remote_post( $url, array(
                    'body'      => $params,
                    'timeout'   => $timeout,
                ) );
                break;
        }

        // If an error occured, return it now
        if ( is_wp_error( $result ) ) {
            return $result;
        }

        // If the HTTP code isn't 200, something went wrong
        if ( $result['response']['code'] != 200 ) {
            // Decode error message
            $body = json_decode( $result['body'] );

            // Define the error message
            $message = array();
            if ( isset( $body->error ) ) {
                $message[] = $body->error;
            }
            if ( isset( $body->message ) ) {
                $message[] = $body->message;
            }

            // Return WP_Error
            return new WP_Error( 
                $result['response']['code'], 
                'Buffer API Error: HTTP Code ' . $result['response']['code'] . '. #' . $body->code . ' - ' . implode( "\n", $message ) 
            );
        }

        // All OK, return response
        return json_decode( $result['body'] );

    }

    /**
     * Performs POST and GET requests through PHP's curl_exec() function.
     *
     * If this function is called, request_wordpress() failed, most likely
     * due to a DNS lookup failure or CloudFlare failing to respond.
     *
     * We therefore use CURLOPT_RESOLVE, to tell cURL the IP address for the domain.
     *
     * @since   3.2.6
     *
     * @param   string  $url        URL
     * @param   string  $method     Method (post|get)
     * @param   array   $params     Parameters
     * @param   int     $timeout    Timeout, in seconds (default: 10)
     * @return  mixed               WP_Error | object
     */
    private function request_curl( $url, $method, $params, $timeout = 10 ) {

        // Init
        $ch = curl_init();

        // Set request specific options
        switch ( $method ) {
            /**
             * GET
             */
            case 'get':
                curl_setopt_array( $ch, array(
                    CURLOPT_URL             => $url . '&' . http_build_query( $params ),
                    CURLOPT_RESOLVE         => array( 
                        str_replace( 'https://', '', $this->api_endpoint ) . ':443:104.16.97.40',
                        str_replace( 'https://', '', $this->api_endpoint ) . ':443:104.16.98.40',
                    ),
                ) );
                break;

            /**
             * POST
             */
            case 'post':
                curl_setopt_array( $ch, array(
                    CURLOPT_URL             => $url,
                    CURLOPT_POST            => true,
                    CURLOPT_POSTFIELDS      => http_build_query( $params ),
                    CURLOPT_RESOLVE         => array( 
                        str_replace( 'https://', '', $this->api_endpoint ) . ':443:104.16.97.40',
                        str_replace( 'https://', '', $this->api_endpoint ) . ':443:104.16.98.40',
                    ),
                ) );
                break;
        }

        // Set shared options
        curl_setopt_array( $ch, array(
            CURLOPT_RETURNTRANSFER  => true,
            CURLOPT_HEADER          => false,
            CURLOPT_FOLLOWLOCATION  => true,
            CURLOPT_MAXREDIRS       => 10,
            CURLOPT_CONNECTTIMEOUT  => $timeout,
            CURLOPT_TIMEOUT         => $timeout,
        ) );

        // Execute
        $result     = curl_exec( $ch );
        $http_code  = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
        $error      = curl_error( $ch );
        curl_close( $ch );

        // If our error string isn't empty, something went wrong
        if ( ! empty( $error ) ) {
            return new WP_Error( $this->base->plugin->name . '_api_request_curl', $error );
        }

        // If HTTP code isn't 200, something went wrong
        if ( $http_code != 200 ) {
            // Decode error message
            $result = json_decode( $result );

            // Return basic WP_Error if we don't have any more information
            if ( is_null( $result ) ) {
                return new WP_Error(
                    $http_code,
                    'Buffer API Error: HTTP Code ' . $http_code . '. Sorry, we don\'t have any more information about this error. Please try again.'
                );
            }

            // Define the error message
            $message = array();
            if ( isset( $result->error ) ) {
                $message[] = $result->error;
            }
            if ( isset( $result->message ) ) {
                $message[] = $result->message;
            }

            // Return WP_Error
            return new WP_Error( $http_code, 'Buffer API Error: HTTP Code ' . $http_code . '. #' . $result->code. ' - ' . implode( "\n", $message )  );
        }
        
        // All OK, return response
        return json_decode( $result );

    }

}