<?php
/**
 * Keywords class
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.0.0
 */
class Page_Generator_Pro_Keywords {

    /**
     * Holds the class object.
     *
     * @since   1.1.3
     *
     * @var     object
     */
    public static $instance;

	/**
	 * Primary SQL Table
     *
     * @since   1.0.0
     *
     * @var     string
	 */
	public $table = 'page_generator_keywords';
	
	/**
	 * Primary SQL Table Primary Key
     *
     * @since   1.0.0
     *
     * @var     string
	 */
	public $key = 'keywordID';

    /**
     * Activation routines for this Model
     *
     * @since   1.0.7
     *
     * @global  $wpdb   WordPress DB Object
     */
    public function activate() {

        global $wpdb;

        // Enable error output if WP_DEBUG is enabled.
        $wpdb->show_errors = true;

        // Create database tables
        $wpdb->query( " CREATE TABLE IF NOT EXISTS " . $wpdb->prefix . "page_generator_keywords (
                            `keywordID` int(10) NOT NULL AUTO_INCREMENT,
                            `keyword` varchar(200) NOT NULL,
                            `data` mediumtext NOT NULL,
                            PRIMARY KEY `keywordID` (`keywordID`),
                            UNIQUE KEY `keyword` (`keyword`)
                        ) ENGINE=MyISAM 
                        DEFAULT CHARSET=" . $wpdb->charset . "
                        AUTO_INCREMENT=1" ); 

    }

    /**
     * Gets a record by its ID
     *
     * @since   1.0.0
     *
     * @param   int    $id  ID
     * @return  mixed       Record | false
     */
    public function get_by_id( $id ) {

        global $wpdb;
       
        // Get record
        $query = $wpdb->prepare("   SELECT *
                                    FROM " . $wpdb->prefix . $this->table . "
                                    WHERE " . $this->key . " = %d
                                    LIMIT 1",
                                    $id ); 
        $results = $wpdb->get_results( $query, ARRAY_A );
        
        // Check a record was found     
        if ( ! $results ) {
            return false;
        }             
        if ( count( $results ) == 0 ) {
            return false;
        }

        // Get single result from array
        $result = $results[0];

        // Stripslashes
        $result['data'] = stripslashes( $result['data'] );

        // Expand data into array
        $result['dataArr'] = explode( "\n", $result['data'] );

        // Return record
        return $result;

    }
    
    /**
     * Gets all results by the key/value pair
     *
     * @since   1.0.0
     *
     * @param   string  $field  Field Name
     * @param   string  $value  Field Value
     * @return  array           Records
     */
    public function get_by( $field, $value ) {
        
        global $wpdb;
       
        // Get record
        $query = $wpdb->prepare("   SELECT *
                                    FROM " . $wpdb->prefix . $this->table . "
                                    WHERE " . $field . " = '%s'",
                                    $value ); 
        $results = $wpdb->get_results( $query, ARRAY_A );

        // Check a record was found     
        if ( ! $results ) {
            return false;
        }             
        if ( count( $results ) == 0 ) {
            return false;
        }

        // Get single result from array
        $result = $results[0];

        // Stripslashes
        $result['data'] = stripslashes( $result['data'] );

        // Expand data into array
        $result['dataArr'] = explode( "\n", $result['data'] );

        // Return
        return $result;

    }
	
    /**
     * Returns an array of records
     *
     * @since   1.0.0
     * 
     * @param   string  $order_by           Order By Column (default: keyword, optional)
     * @param   string  $order              Order Direction (default: ASC, optional)
     * @param   int     $paged              Pagination (default: 1, optional)
     * @param   int     $results_per_page   Results per page (default: 10, optional)
     * @param   string  $search             Search Keywords (optional)
     * @return  array                       Records
     */
    public function get_all( $order_by = 'keyword', $order = 'ASC', $paged = 1, $results_per_page = 10, $search = '' ) {
        
        global $wpdb;
        
        $get_all = ( ( $paged == -1 ) ? true : false );

       	// Search? 
        if ( ! empty( $search ) ) {
	    	$query = $wpdb->prepare( " 	SELECT *
                                    	FROM " . $wpdb->prefix . $this->table . "
                                    	WHERE keyword LIKE '%%%s%%'
                                    	ORDER BY %s %s",
                                    	$search,
                                    	$order_by,
                                    	$order );
        } else {
	        $query = $wpdb->prepare( " 	SELECT *
                                    	FROM " . $wpdb->prefix . $this->table . "
                                    	ORDER BY %s %s",
                                    	$order_by,
                                    	$order );
        }

        // Add Limit
        if ( ! $get_all ) {
            $query = $query . $wpdb->prepare( " LIMIT %d, %d",
                                                ( ( $paged - 1 ) * $results_per_page ),
                                                $results_per_page );
        }

        // Get results
        $results = $wpdb->get_results( $query );

        // Check a record was found     
        if ( ! $results ) {
            return false;
        }             
        if ( count( $results ) == 0 ) {
            return false;
        }

      	return stripslashes_deep( $results );

    }

    /**
     * Confirms whether a keyword already exists.
     *
     * @since   1.0.0
     *
     * @param   string  $keyword    Keyword
     * @return  bool                Exists
     */
    public function exists( $keyword ) {
        
        global $wpdb;
       
        // Get record
        $query = $wpdb->prepare("   SELECT keywordID
                                    FROM " . $wpdb->prefix . $this->table . "
                                    WHERE keyword = '%s'",
                                    $keyword ); 
        $results = $wpdb->get_results( $query, ARRAY_A );

        // Check a record was found     
        if ( ! $results ) {
            return false;
        }             
        if ( count( $results ) == 0 ) {
            return false;
        }

        return true;

    }
    
    /**
     * Get the number of matching records
     *
     * @since   1.0.0
     *
     * @param   string  $search Search Keywords (optional)
     * @return  bool            Exists
     */
    public function total( $search = '' ) {

        global $wpdb;
        
        // Prepare query
        if ( ! empty( $search ) ) {
            $query = $wpdb->prepare( "  SELECT COUNT(" . $this->key . ")
                                        FROM " . $wpdb->prefix . $this->table . "
                                        WHERE keyword LIKE '%%%s%%'",
                                        $search ); 
        } else {
            $query = "  SELECT COUNT( " . $this->key . " )
                        FROM " . $wpdb->prefix . $this->table; 
    
        }
        
        // Return count
        return $wpdb->get_var( $query );

    }

    /**
     * Converts an uploaded text file of keyword data into a data string that can be inserted into the DB
     *
     * @since   1.0.7
     *
     * @param   string  $existing_data  Existing Data
     * @return  string                  New Data
     */ 
    public function import_text_file_data( $existing_data = '' ) {

        // Check a file has been uploaded
        if ( ! isset( $_FILES['file'] ) ) {
            return $existing_data;
        }

        // Check uploaded file is a supported filetype
        if ( ! ( ! empty( $_FILES['file']['type'] ) && preg_match( '/(text|txt)$/i', $_FILES['file']['type'] ) ) && 
            ! preg_match( '/(text|txt)$/i', $_FILES['file']['name'] ) ) {
            return $existing_data;
        }
            
        // Get file contents
        $handle = fopen( $_FILES['file']['tmp_name'], 'r' );
        $contents = fread( $handle, filesize( $_FILES['file']['tmp_name'] ) );
        fclose( $handle );
                
        // Add / append data
        $existing_data .= ( ( strlen( $existing_data ) > 0 ) ? "\n" . $contents : $contents );  

        // Return
        return $existing_data;

    }

    /**
     * Converts an uploaded CSV file of keywords data into an array of keywords and terms, that can then be added
     * to the keywords DB table
     *
     * @since   1.7.3
     *
     * @param   string  $keywords_location  Keywords Location in CSV File (columns|rows)
     *
     * @return  mixed   WP_Error | int
     */ 
    public function import_csv_file_data( $keywords_location = 'columns' ) {

        // Check a file has been uploaded
        if ( ! isset( $_FILES['file'] ) ) {
            return new WP_Error( 'page_generator_pro_keywords_import_csv_file_data_missing', __( 'No file was uploaded.', 'page-generator-pro' ) );
        }

        // Check uploaded file is a supported filetype
        if ( ! ( ! empty( $_FILES['file']['type'] ) && preg_match( '/(csv)$/i', $_FILES['file']['type'] ) ) && 
            ! preg_match( '/(csv)$/i', $_FILES['file']['name'] ) ) {
            return new WP_Error( 'page_generator_pro_keywords_import_csv_file_data_unsupported_file_tye', __( 'The file uploaded is not a supported file type.  Please ensure you are uploading a CSV file.', 'page-generator-pro' ) );
        }
            
        // Get file contents
        $handle = fopen( $_FILES['file']['tmp_name'], 'r' );
        $contents = trim( fread( $handle, filesize( $_FILES['file']['tmp_name'] ) ) );
        fclose( $handle );

        // Bail if file contents are empty
        if ( strlen( $contents ) == 0 || empty( $contents ) ) {
            return new WP_Error( 'page_generator_pro_keywords_import_csv_file_data_empty', __( 'The uploaded file contains no data.', 'page-generator-pro' ) );  
        }

        // Fetch rows
        $rows = explode( "\n", $contents );

        // Bail if no rows found
        if ( count( $rows ) < 2 ) {
            return new WP_Error( 'page_generator_pro_keywords_import_csv_file_data_no_rows', __( 'The uploaded file only contains one row of data.  There must be at least two rows; the first being the keywords.', 'page-generator-pro' ) );
        }
        
        // Build array comprising of keywords and their terms
        $keywords = array();
        $keywords_terms = array();
        foreach ( $rows as $index => $row ) {
            $terms = str_getcsv( $row );

            // Depending on where the keywords are, parse the terms
            switch ( $keywords_location ) {
                /**
                 * Columns
                 */
                case 'columns':
                    // First row are keywords
                    if ( $index == 0 ) {
                        foreach ( $terms as $term ) {
                            $keywords[] = trim( $term );
                        }
                        continue;
                    }

                    // Add this row's terms to the keywords array
                    foreach ( $terms as $term_index => $term ) {
                        if ( ! is_array( $keywords_terms[ $keywords[ $term_index ] ]  ) ) {
                            $keywords_terms[ $keywords[ $term_index ] ] = array();
                        }

                        $keywords_terms[ $keywords[ $term_index ] ][] = trim( $term );
                    }
                    break;

                /**
                 * Rows
                 */
                case 'rows':
                    // First term is a keyword; all other terms are the keyword's terms
                    // Add this row's terms to the keywords array
                    foreach ( $terms as $term_index => $term ) {
                        // First term is the keyword
                        if ( $term_index == 0 ) {
                            $keyword = trim( $term );
                            $keywords[] = $keyword;
                            continue;
                        }

                        // Other terms are terms
                        if ( ! is_array( $keywords_terms[ $keyword ]  ) ) {
                            $keywords_terms[ $keyword ] = array();
                        }

                        $keywords_terms[ $keyword ][] = trim( $term );
                    }
                    break;
            }
        }

        // Bail if we couldn't get any keyword terms
        if ( empty( $keywords_terms ) || count( $keywords_terms ) == 0 ) {
            return new WP_Error( 'page_generator_pro_keywords_import_csv_file_data_no_keyword_terms', __( 'No keywords and/or terms could be found in the uploaded file.', 'page-generator-pro' ) );
        }

        // For each keyword, check that a keyword doesn't already exist in the database
        foreach ( $keywords as $keyword ) {
            $exists = $this->exists( $keyword );

            // If the keyword exists, exit
            if ( $exists ) {
                return new WP_Error( 'page_generator_pro_keywords_import_csv_file_keyword_exists', sprintf( __( 'The %s keyword already exists.  No keywords or terms were imported.', 'page-generator-pro' ), $keyword ) );
            }
        }

        // If here, we are OK to add keywords and their terms to the database
        // Iterate through keyword terms, adding them to the database
        foreach ( $keywords_terms as $keyword => $terms ) {
            $result = $this->save( array(
                'keyword' => $keyword,
                'data'    => implode( "\n", $terms ),
            ) );

            // If an error occured, bail
            if ( is_wp_error( $result ) ) {
                return $result;
            }
        }

        // Return the number of keywords added
        return count( $keywords_terms );

    }

    /**
     * Adds or edits a record, based on the given data array.
     *
     * @since   1.0.0
     * 
     * @param   array   $data   Array of data to save
     * @param   int     $id     ID (if set, edits the existing record)
     * @return  mixed           ID or WP_Error
     */
    public function save( $data, $id = '' ) {

        global $wpdb;

        // Check for required data fields
        if ( empty( $data['keyword'] ) ) {
            return new WP_Error( 'page_generator_pro_keywords_save_validation_error', __( 'Please complete the keyword field.', 'page-generator-pro' ) );
        }
        if ( empty( $data['data'] ) ) {
            return new WP_Error( 'page_generator_pro_keywords_save_validation_error', __( 'Please complete the keyword data field.', 'page-generator-pro' ) );
        }

        // Ensure that the keyword does not contain spaces
        if ( ! preg_match( '/^[a-zA-Z0-9_]+$/', $data['keyword'] ) ) {
            return new WP_Error( 'page_generator_pro_keywords_save_validation_error', __( 'The Keyword field can only contain letters, numbers and underscores.', 'page-generator-pro' ) );
        }

        if ( ! empty( $id ) ) {
            // Editing an existing record
            // Build query
            $query = array();
            foreach ( $data as $key => $value ) {
                switch ($key) {
                    case $this->key:
                        // Ignore
                        break;
                    default:
                        // String
                        $query[] = $key." = '" . htmlentities( $value, ENT_QUOTES, 'UTF-8' ) . "'";
                        break;
                }
            }
            
            // Prepare query to an SQL string
            $query = $wpdb->prepare("   UPDATE " . $wpdb->prefix . $this->table . "
                                        SET " . implode( ',', $query ) . "
                                        WHERE " . $this->key . " = %s",
                                        $id );
                                        
            // Run query
            $result = $wpdb->query( $query );

            // Check query was successful
            if ($result === FALSE) {
                return new WP_Error( 'db_query_error', __('Keyword could not be edited in the database. DB said: ' . $wpdb->last_error ), $wpdb->last_error ); 
            }

            // Success!
            return $id; 
        } else {
            // Adding a new record  
            $query = $wpdb->prepare( "  INSERT INTO " . $wpdb->prefix . $this->table . " (keyword, data)
                                        VALUES (%s, %s)",
                                        $data['keyword'],
                                        $data['data'] );
                                    
            // Run query
            $result = $wpdb->query( $query );
          
            // Check query was successful
            if ( $result === FALSE ) {
                return new WP_Error( 'db_query_error', __( 'Keyword could not be added to the database. DB said: '.$wpdb->last_error ), $wpdb->last_error ); 
            }
            
            // Get and return ID
            return $wpdb->insert_id;
        }    

    }
 
    /**
     * Deletes the record for the given primary key ID
     *
     * @since   1.0.0
     * 
     * @param   mixed   $data   Single ID or array of IDs
     * @return  bool            Success
     */
    public function delete( $data ) {

        global $wpdb;
        
        // Run query
        if ( is_array( $data ) ) {
            $query = "  DELETE FROM " . $wpdb->prefix . $this->table . "
                        WHERE " . $this->key . " IN (" . implode( ',', $data ) . ")";
        } else {
            $query = $wpdb->prepare("   DELETE FROM " . $wpdb->prefix . $this->table . "
                                        WHERE " . $this->key . " = %s
                                        LIMIT 1",
                                        $data );
        }
        $result = $wpdb->query( $query );
                          
        // Check query was successful
        if ( $result === FALSE ) {
            return new WP_Error( 'db_query_error', __( 'Record(s) could not be deleted from the database. DB said: '.$wpdb->last_error ), $wpdb->last_error );
        }

        return true;

    }
    
    /**
     * Changes the given field's old value to a new value
     *
     * @since   1.0.0
     *
     * @param   string $field       Field
     * @param   string $old_value   Old Value
     * @param   string $new_value   New Value
     * @return  bool                Success
     */
    public function change( $field, $old_value, $new_value ) {
        
        global $wpdb;
        
        // Prepare query to an SQL string
        $query = $wpdb->prepare( "  UPDATE " . $wpdb->prefix . $this->table . "
                                    SET " . $field . " = %s
                                    WHERE " . $field . " = %s",
                                    $new_value,
                                    $old_value );
        $result = $wpdb->query( $query );

        // Check query was successful
        if ( $result === FALSE ) {
            return new WP_Error( 'db_query_error', __( 'Record\'s field could not be changed in the database. DB said: '.$wpdb->last_error ), $wpdb->last_error ); 
        }
        
        return true;  
        
    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.3
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}