<?php
/**
 * Importer class
 * 
 * @package Page Generator Pro
 * @author  Tim Carr
 * @version 1.1.8
 */
class Page_Generator_Pro_Import {

    /**
     * Holds the class object.
     *
     * @since   1.1.8
     *
     * @var     object
     */
    public static $instance;

    /**
     * Imports a remote image into the WordPress Media Library
     *
     * @since   1.1.8
     *
     * @param   string  $source     Source URL
     * @param   int     $post_id    Post ID
     * @param   string  $title      Image Title (optional)
     * @param   string  $caption    Image Caption (optional)
     * @return  mixed               Image ID | WP_Error
     */
    public function import_remote_image( $source, $post_id, $title = '', $caption = '' ) {

        // Import the remote image
        if ( ! function_exists( 'media_handle_upload' ) ) {
            require_once( ABSPATH . 'wp-admin/includes/image.php' );
            require_once( ABSPATH . 'wp-admin/includes/file.php' );
            require_once( ABSPATH . 'wp-admin/includes/media.php' );
        }

        // Get the remote image
        $tmp = download_url( $source );
        if ( is_wp_error( $tmp ) ) {
            return $tmp;
        }

        // Get image type
        $type = getimagesize( $tmp );
        if ( ! isset( $type['mime'] ) ) {
            echo 'MIME Error';
            die();

            return new WP_Error( __( 'Could not identify MIME type of imported image.', 'page-generator-pro' ) );
        }
        list( $type, $ext ) = explode( '/', $type['mime'] );
        unset( $type );

        // Define image filename
        $file_array['name']     = basename( $source );
        $file_array['tmp_name'] = $tmp;

        // Add the extension to the filename if it doesn't exist
        // This happens if we streamed an image URL e.g. http://placehold.it/400x400
        if ( strpos( $file_array['name'], '.' . $ext ) === false ) {
            $file_array['name'] .= '.' . $ext;
        }

        // Import the image into the Media Library
        $image_id = media_handle_sideload( $file_array, $post_id, '' );
        if ( is_wp_error( $image_id ) ) {
            return $image_id;
        }

        // If a title or caption has been defined, set the now
        if ( ! empty( $title ) || ! empty( $caption ) ) {
            $attachment = get_post( $image_id );
            wp_update_post( array(
                'ID'            => $image_id,
                'post_title'    => sanitize_text_field( $title ),
                'post_content'  => sanitize_text_field( $caption ),
            ) );
        }

        // Return the image ID (or if an error occured, this will be a WP_Error instance)
        return $image_id;

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.8
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}