<?php
/**
 * Handles creating, editing, deleting and calling the generate routine
 * for the Generate Content section of the Plugin.
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.2.1
 */
class Page_Generator_Pro_Groups {

    /**
     * Holds the class object.
     *
     * @since   1.2.1
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the base class object.
     *
     * @since   1.2.3
     *
     * @var     object
     */
    public $base;

    /**
     * Holds the common class object.
     *
     * @since   1.2.3
     *
     * @var     object
     */
    public $common;

    /**
     * Stores Keywords available to the Group
     *
     * @since   1.2.3
     *
     * @var     array
     */
    public $keywords;

    /**
     * Stores a Group's settings
     *
     * @since   1.2.3
     *
     * @var     array
     */
    public $settings;

    /**
     * Stores success and error messages
     *
     * @since 1.2.3
     *
     * @var array
     */
    public $notices = array(
        'success'   => array(),
        'error'     => array(),
    );

    /**
     * Constructor
     *
     * @since 1.2.3
     */
    public function __construct() {

        // Process any notices that need to be displayed
        add_action( 'admin_notices', array( $this, 'admin_notices' ) );

        // WP_List_Table Columns
        add_filter( 'manage_edit-page-generator-pro_columns', array( $this, 'admin_columns' ) );
        add_action( 'manage_page-generator-pro_posts_custom_column', array( $this, 'admin_columns_output' ), 10, 2 );

        // WP_List_Table Row Actions
        add_filter( 'post_row_actions', array( $this, 'admin_row_actions' ) );

        // Run any row actions called from the WP_List_Table
        add_action( 'init', array( $this, 'run_row_actions' ) );

        // Modify Post Messages
        add_filter( 'post_updated_messages', array( $this, 'post_updated_messages' ) );

        // Before Title
        add_action( 'edit_form_top', array( $this, 'output_keywords_dropdown_before_title' ) );

        // Meta Boxes
        add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );

        // Save Group
        add_action( 'save_post', array( $this, 'save_post' ) );
        add_filter( 'wpseo_sanitize_post_meta__yoast_wpseo_canonical', array( $this, 'wpseo_sanitize_post_meta__yoast_wpseo_canonical' ), 10, 4 );
       // add_action( 'wpseo_saved_postdata', array( $this, 'wpseo_saved_postdata' ) );

        // Page Generator
        if ( class_exists( 'Page_Generator' ) ) {
            add_action( 'init', array( $this, 'limit_admin' ) );
            add_filter( 'wp_insert_post_empty_content', array( $this, 'limit_xml_rpc' ), 10, 2 );
        }

    }

    /**
     * Creates a single Group, if none exist, when the Plugin is activated.
     *
     * @since   1.3.8
     *
     * @global  $wpdb   WordPress DB Object
     */
    public function activate() {

        // Bail if we already have at least one Group
        $number_of_groups = $this->get_count();
        if ( $number_of_groups > 0 ) {
            return;
        }

        // Create Group
        wp_insert_post( array(
            'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
            'post_status'   => 'publish',
            'post_title'    => __( 'Title' ),
            'post_content'  => __( 'Edit this content, replacing it with the content you want to generate. You can use {keywords} here too.  Need help? Visit <a href="https://www.wpzinc.com/documentation/page-generator-pro/generate/" target="_blank">https://www.wpzinc.com/documentation/page-generator-pro/generate/</a>' ),
        ) );

    }

    /**
     * Checks the transient to see if any admin notices need to be output now
     * i.e. if a Test or Delete call was made on this Group.
     *
     * @since   1.2.3
     */
    public function admin_notices() {

        global $post;

        // Don't do anything if we're not on this Plugin's CPT
        if ( empty( $post ) ) {
            return;
        }
        if ( get_post_type( $post ) != Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
            return;
        }

        // Get current user
        $user = wp_get_current_user();
        
        // Check the transient for any notices
        $notices = get_transient( 'page_generator_pro_groups_notices_' . $post->ID . '_' . $user->ID );
        if ( empty( $notices ) ) {
            return;
        }

        // If here, notice(s) exist
        // Store them in the class variable and delete the transient
        if ( isset( $notices['success'] ) ) {
            $this->notices['success'] = $notices['success'];
        }
        if ( isset( $notices['error'] ) ) {
            $this->notices['error'] = $notices['error'];
        }
        delete_transient( 'page_generator_pro_groups_notices_' . $post->ID . '_' . $user->ID );

        // Output success notice(s)
        if ( is_array( $this->notices['success'] ) ) {
            foreach ( $this->notices['success'] as $message ) {
                ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo $message; ?></p>
                </div>
                <?php
            }
        }

        // Output error notice(s)
        if ( is_array( $this->notices['error'] ) ) {
            foreach ( $this->notices['error'] as $message ) {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo $message; ?></p>
                </div>
                <?php
            }
        }

    }

    /**
     * Adds columns to the Groups CPT within the WordPress Administration List Table
     * 
     * @since   1.2.3
     *
     * @param   array   $columns    Columns
     * @return  array               New Columns
     */
    public function admin_columns( $columns ) {

        // Inject columns
        $new_columns = array(
            'cb'                    => '<input type="checkbox" />',
            'title'                 => __( 'Title' ),
            'type'                  => __( 'Content Type', $this->base->plugin->name ),
            'generated_count'       => __( 'No. Generated Items', $this->base->plugin->name ),
            'date'                  => __( 'Date' ),
        );

        // Filter columns
        $new_columns = apply_filters( 'page_generator_pro_admin_admin_columns', $new_columns, $columns );

        return $new_columns;

    }


    /**
     * Manages the data to be displayed within a column on the Groups CPT within 
     * the WordPress Administration List Table
     * 
     * @since   1.2.3
     *
     * @param   string  $column_name    Column Name
     * @param   int     $post_id        Group Post ID
     */
    public function admin_columns_output( $column_name, $post_id ) {

        switch ( $column_name ) {
            /**
             * Type
             */
            case 'type':
                // Get group settings
                $settings = $this->get_settings( $post_id );
                if ( isset( $settings['type'] ) ) {
                    echo $settings['type'];
                } else {
                    echo __( 'N/A', $this->base->plugin->name );
                }
                break;

            /**
             * Number of Generated Pages
             */
            case 'generated_count':
                // Get group settings
                $settings = $this->get_settings( $post_id );
                if ( isset( $settings['generated_pages_count'] ) ) {
                    echo $settings['generated_pages_count'];
                } else {
                    echo __( 'N/A', $this->base->plugin->name );
                }
                break;

            default:
                do_action( 'page_generator_pro_admin_admin_columns_output', $column_name, $post_id );
                break;
        }

    }

    /**
     * Adds Duplicate and Generate Row Actions to each Groups CPT within
     * the WordPress Administration List Table
     *
     * @since   1.2.3
     *
     * @param   array   $actions    Row Actions
     * @return  array               Row Actions
     */
    public function admin_row_actions( $actions ) {

        global $post;

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Get instances
        $post_type_instance = Page_Generator_Pro_PostType::get_instance();

        // Bail if not a Groups CPT
        if ( get_post_type( $post ) != $post_type_instance->post_type_name ) {
            return $actions;
        }

        // Add Duplicate and Generate Actions
        $actions['duplicate'] = '<a href="edit.php?post_type=' . $post_type_instance->post_type_name . '&' . $this->base->plugin->name . '-action=duplicate&id=' . $post->ID . '">' . __( 'Duplicate', $this->base->plugin->name ) . '</a>';
        $actions['generate'] = '<a href="admin.php?page=' . $this->base->plugin->name . '-generate&id=' . $post->ID . '&type=content">' . __( 'Generate', $this->base->plugin->name ) . '</a>';

        // Filter actions
        $actions = apply_filters( 'page_generator_pro_admin_admin_row_actions', $actions, $post );

        // Return
        return $actions;

    }

    /**
     * Checks if the Duplicate row action was clicked by the User, and if so
     * duplicates the given Group ID.
     *
     * @since   1.2.3
     */
    public function run_row_actions() {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // If no action specified, return
        if ( ! isset( $_REQUEST[ $this->base->plugin->name . '-action' ] ) ) {
            return;
        }

        // Depending on the action, do something
        $action = sanitize_text_field( $_REQUEST[ $this->base->plugin->name . '-action' ] );
        switch ( $action ) {

            /**
             * Duplicate
             */
            case 'duplicate':
                $id = absint( $_REQUEST['id'] );
                $this->duplicate( $id );
                break;

            default:
                do_action( 'page_generator_pro_admin_run_row_actions', $action );
                break;

        }

    }

    /**
     * Defines admin notices for the Post Type.
     *
     * This also removes the 'View post' link on the message, which would result
     * in an error on the frontend.
     *
     * @since   1.5.8
     *
     * @param   array   $messages   Messages
     * @return  array               Messages
     */
    public function post_updated_messages( $messages ) {

        $messages[ Page_Generator_Pro_PostType::get_instance()->post_type_name ] = array(
            0 => '', // Unused. Messages start at index 1.
            1 => __( 'Group updated.', 'page-generator-pro' ),
            2 => __( 'Custom field updated.', 'page-generator-pro'  ),
            3 => __( 'Custom field deleted.', 'page-generator-pro'  ),
            4 => __( 'Group updated.', 'page-generator-pro'  ),
            5 => ( isset( $_GET['revision'] ) ? sprintf( __( 'Group restored to revision from %s.', 'page-generator-pro'  ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false ),
            6 => __( 'Group published.', 'page-generator-pro'  ),
            7 => __( 'Group saved.', 'page-generator-pro'  ),
            8 => __( 'Group submitted.', 'page-generator-pro'  ),
            9 => __( 'Group scheduled.', 'page-generator-pro'  ),
            10 => __( 'Group draft updated.', 'page-generator-pro'  ),
        );

        return $messages;

    }

    /**
     * Outputs the Keywords Dropdown before the Title field
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_keywords_dropdown_before_title( $post ) {

        // Don't do anything if we're not on this Plugin's CPT
        if ( get_post_type( $post ) !== Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
            return;
        }

        // Get all available keywords, post types, taxonomies, authors and other settings that we might use on the admin screen
        $this->keywords = Page_Generator_Pro_Keywords::get_instance()->get_all( 'keyword', 'ASC', -1 );

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-title-keywords.php' ); 

    }

    /**
     * Registers meta boxes for the Generate Custom Post Type
     *
     * @since 1.2.3
     */
    public function add_meta_boxes() {

        // Get instances
        $this->base   = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );
        $this->common = Page_Generator_Pro_Common::get_instance();
        $post_type_instance = Page_Generator_Pro_PostType::get_instance();

        // Remove all metaboxes
        $this->remove_all_meta_boxes();

        // Permalink
        add_meta_box( 
            $post_type_instance->post_type_name . '-permalink', 
            __( 'Permalink', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_permalink' ), 
            $post_type_instance->post_type_name,
            'normal' 
        );
        
        // Excerpt
        add_meta_box( 
            $post_type_instance->post_type_name . '-excerpt', 
            __( 'Excerpt', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_excerpt' ), 
            $post_type_instance->post_type_name,
            'normal'  
        );

        // Custom Fields
        add_meta_box( 
            $post_type_instance->post_type_name . '-custom-fields', 
            __( 'Custom Fields', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_custom_fields' ), 
            $post_type_instance->post_type_name,
            'normal'  
        );

        // Author
        add_meta_box( 
            $post_type_instance->post_type_name . '-author', 
            __( 'Author', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_author' ), 
            $post_type_instance->post_type_name,
            'normal'  
        );

        // Discussion
        add_meta_box( 
            $post_type_instance->post_type_name . '-discussion', 
            __( 'Discussion', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_discussion' ), 
            $post_type_instance->post_type_name,
            'normal'  
        );

        // Upgrade
        if ( class_exists( 'Page_Generator' ) ) {
            add_meta_box( 
                $post_type_instance->post_type_name . '-upgrade', 
                __( 'Upgrade', $this->base->plugin->name ), 
                array( $this, 'output_meta_box_upgrade' ), 
                $post_type_instance->post_type_name,
                'normal'  
            );
        }

        /**
         * Sidebar
         */

        // Page Builders
        do_action( 'page_generator_pro_groups_add_meta_boxes', $post_type_instance );
        
        // Actions Top
        add_meta_box( 
            $post_type_instance->post_type_name . '-actions', 
            __( 'Actions', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_actions' ), 
            $post_type_instance->post_type_name,
            'side'
        );

        // Publish
        add_meta_box( 
            $post_type_instance->post_type_name . '-publish', 
            __( 'Publish', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_publish' ), 
            $post_type_instance->post_type_name,
            'side'
        );

        // Generation
        add_meta_box( 
            $post_type_instance->post_type_name . '-generation', 
            __( 'Generation', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_generation' ), 
            $post_type_instance->post_type_name,
            'side'
        );

        // Attributes
        add_meta_box( 
            $post_type_instance->post_type_name . '-attributes', 
            __( 'Attributes', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_attributes' ), 
            $post_type_instance->post_type_name,
            'side'
        );

        // Taxonomies
        add_meta_box( 
            $post_type_instance->post_type_name . '-taxonomies', 
            __( 'Taxonomies', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_taxonomies' ), 
            $post_type_instance->post_type_name,
            'side'
        );

        // Featured Image
        add_meta_box( 
            $post_type_instance->post_type_name . '-featured-image', 
            __( 'Featured Image', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_featured_image' ), 
            $post_type_instance->post_type_name,
            'side'
        );

        // Actions Bottom
        add_meta_box( 
            $post_type_instance->post_type_name . '-actions-bottom', 
            __( 'Actions', $this->base->plugin->name ), 
            array( $this, 'output_meta_box_actions' ), 
            $post_type_instance->post_type_name,
            'side',
            'low'
        );

    }

    /**
     * Defines the meta boxes which are permitted for display on the Groups screen.
     *
     * @since   1.2.3
     *
     * @return  array   Permitted Meta Boxes
     */
    public function permitted_meta_boxes() {

        // Define permitted meta boxes
        $permitted_meta_boxes = array(
            // Avia Builder
            'avia_builder',
            'layout',
            'preview',
            'hierarchy',

            // BeTheme / Muffin Page Builder
            'mfn-meta-page-generator-pro',

            // Visual Composer
            'wpb_visual_composer',

            // AIOSEO
            'aiosp',

            // Yoast SEO
            'wpseo_meta',
        );

        // Filter permitted meta boxes
        $permitted_meta_boxes = apply_filters( 'page_generator_pro_groups_permitted_meta_boxes', $permitted_meta_boxes );

        return $permitted_meta_boxes;

    }

    /**
     * Removes metaboxes added by most other Plugins and WordPress, so we can contrl
     * the UI better.
     *
     * @since   1.2.3
     *
     * @global  array   $wp_meta_boxes  Array of registered metaboxes.
     */
    public function remove_all_meta_boxes() {

        global $wp_meta_boxes;

        // Get permitted meta boxes
        $permitted_meta_boxes = $this->permitted_meta_boxes();

        // Bail if no meta boxes for this CPT exist
        if ( ! isset( $wp_meta_boxes['page-generator-pro'] ) ) {
            return;
        }
        
        // Iterate through all registered meta boxes, removing those that aren't permitted
        foreach ( $wp_meta_boxes['page-generator-pro'] as $position => $contexts ) {
            foreach ( $contexts as $context => $meta_boxes ) {
                foreach ( $meta_boxes as $meta_box_id => $meta_box ) {
                    // If this meta box isn't in the array of permitted meta boxes, remove it now
                    if ( empty( $permitted_meta_boxes ) || ! in_array( $meta_box_id, $permitted_meta_boxes ) ) {
                        unset( $wp_meta_boxes['page-generator-pro'][ $position ][ $context ][ $meta_box_id ] );
                    }
                }
            }
        }

    }

    /**
     * Outputs the Permalink Meta Box
     *
     * @since 1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_permalink( $post ) {

        // Get all available keywords, post types, taxonomies, authors and other settings that we might use on the admin screen
        $this->keywords = Page_Generator_Pro_Keywords::get_instance()->get_all( 'keyword', 'ASC', -1 );

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-permalink.php' ); 

    }

    /**
     * Outputs the Excerpt Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_excerpt( $post ) {

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-excerpt.php' ); 

    }

    /**
     * Outputs the Custom Fields Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_custom_fields( $post ) {

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-custom-fields.php' ); 

    }

    /**
     * Outputs the Author Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_author( $post ) {

        // Get options
        $authors = Page_Generator_Pro_Common::get_instance()->get_authors();

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-author.php' ); 

    }

    /**
     * Outputs the Discussion Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_discussion( $post ) {

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-discussion.php' ); 

    }

    /**
     * Outputs the Upgrade Meta Box
     *
     * @since   1.3.8
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_upgrade( $post ) {

        // Load view
        include( $this->base->plugin->folder . '/_modules/dashboard/views/footer-upgrade-embedded.php' );

    }

    /**
     * Outputs the Divi Settings Sidebar Meta Box
     *
     * @since   1.6.4
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_divi( $post ) {

        // Get settings, as this is the first meta box to load
        $this->settings = $this->get_settings( $post->ID );



        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-actions.php' ); 

    }

    /**
     * Outputs the Actions Sidebar Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_actions( $post ) {

        // Get settings, as this is the first meta box to load
        $this->settings = $this->get_settings( $post->ID );

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-actions.php' ); 

    }

    /**
     * Outputs the Publish Sidebar Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_publish( $post ) {

        // Get options
        $post_types             = $this->common->get_post_types();
        $statuses               = $this->common->get_post_statuses();
        $date_options           = $this->common->get_date_options();
        $schedule_units         = $this->common->get_schedule_units();

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-publish.php' ); 

    }

    /**
     * Outputs the Generation Sidebar Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_generation( $post ) {

        // Get options
        $methods = $this->common->get_methods();
        $overwrite_methods = $this->common->get_overwrite_methods();

        // Define labels
        $labels = array(
            'singular'  => __( 'Post', 'page-generator-pro' ),
            'plural'    => __( 'Posts', 'page-generator-pro' ),
        );

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-generation.php' ); 

    }

    /**
     * Outputs the Attributes Sidebar Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_attributes( $post ) {

        // Get options
        $hierarchical_post_types = $this->common->get_hierarchical_post_types();
        $post_types              = $this->common->get_post_types();
        $post_types_templates    = $this->common->get_post_types_templates();

        // Build string of hierarchal post types to use as a selector class
        $hierarchical_post_types_class = '';
        if ( is_array( $hierarchical_post_types ) && count( $hierarchical_post_types ) > 0 ) {
            foreach ( $hierarchical_post_types as $type => $post_type ) {
                $hierarchical_post_types_class .= $type . ' ';
            }
        }

        // Build string of post types to use as a selector class
        $post_types_class = '';
        if ( is_array( $post_types ) && count( $post_types ) > 0 ) {
            foreach ( $post_types as $type => $post_type ) {
                $post_types_class .= $type . ' ';
            }
        }

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-attributes.php' ); 

    }

    /**
     * Outputs the Taxonomies Sidebar Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_taxonomies( $post ) {

        // Fetch some taxonomy information for our Taxonomy meta boxes
        $post_types             = $this->common->get_post_types();
        $taxonomies             = $this->common->get_taxonomies();
        $ignored_taxonomies     = $this->common->get_excluded_taxonomies();

        // Iterate through taxonomies, outputting options for each
        foreach ( $taxonomies as $taxonomy ) {
            // Skip ignored taxonomies
            if ( in_array( $taxonomy->name, $ignored_taxonomies ) ) {
                continue;
            }

            // Build list of Post Types this taxonomy is registered for use on
            $post_types_string = '';
            foreach ( $taxonomy->object_type as $post_type ) {
                $post_types_string = $post_types_string . $post_type . ' ';
            }

            // Load view
            include( $this->base->plugin->folder . 'views/admin/generate-meta-box-taxonomies.php' ); 
        }

    }

    /**
     * Outputs the Featured Image Sidebar Meta Box
     *
     * @since   1.2.3
     *
     * @param   WP_Post     $post   Custom Post Type's Post
     */
    public function output_meta_box_featured_image( $post ) {

        // Get options
        $featured_image_sources = Page_Generator_Pro_Common::get_instance()->get_featured_image_sources();

        // Load view
        include( $this->base->plugin->folder . 'views/admin/generate-meta-box-featured-image.php' ); 

    }

    /**
     * Defines a default settings structure when creating a new group
     *
     * @since   1.2.0
     *
     * @return  array   Group
     */
    public function get_defaults() {

        // Get Defaults
        $defaults = array(
            'title'         => '',
            'permalink'     => '',
            'content'       => '',
            'excerpt'       => '',
            'meta'          => array(),
            'rotateAuthors' => 0,
            'author'        => '',
            'comments'      => 0,
            'trackbacks'    => 0,
            'type'          => 'page',
            'status'        => 'publish',
            'date_option'   => 'now',
            'date_specific' => date( 'Y-m-d' ),
            'date_min'      => date( 'Y-m-d', strtotime( '-1 week' ) ),
            'date_max'      => date( 'Y-m-d' ),
            'schedule'      => 1,
            'scheduleUnit'  => 'hours',
            'method'        => 'all',
            'overwrite'     => 0,
            'numberOfPosts' => 0,
            'resumeIndex'   => 0,
            'pageParent'    => '',
            'pageTemplate'  => '',
            'tax'           => '',
            'featured_image_source'     => '',
            'featured_image'            => '',
            'featured_image_location'   => '',
            'featured_image_alt'        => '',
        );

        // Allow devs to filter defaults.
        $defaults = apply_filters( 'page_generator_pro_groups_get_defaults', $defaults );

        // Return.
        return $defaults;
        
    }

    /**
     * Returns a Group's Settings by the given Group ID
     *
     * @since   1.2.1
     *
     * @param   int     $id     ID
     * @return  mixed           false | array
     */
    public function get_settings( $id ) {

        // Get settings
        $settings = get_post_meta( $id, '_page_generator_pro_settings', true );

        // If the result isn't an array, we're getting settings for a new Group, so just use the defaults
        if ( ! is_array( $settings ) ) {
            $settings = $this->get_defaults();
        } else {
            // Store the Post's Title and Content in the settings, for backward compat
            $post               = get_post( $id );
            $settings['title']  = $post->post_title;
            $settings['content']= $post->post_content;

            // Merge with defaults, so keys are always set
            $settings = array_merge( $this->get_defaults(), $settings );  
        }

        // Fetch all Metadata stored against the Group ID, and add that to the settings array
        $settings['post_meta'] = $this->get_post_meta( $id );

        // Add the generated pages count
        $settings['generated_pages_count'] = $this->get_generated_count_by_id( $id );
        
        // Return settings
        return $settings;

    }

    /**
     * Returns all Post Metadata for the given Group ID, excluding some specific keys.
     *
     * This ensures that Page Builder data, ACF data etc. is included in the Group
     * settings and subsequently copied to the generated Page.
     *
     * @since   1.4.4
     *
     * @param   int     $id     Group ID
     * @return  array           Metadata
     */
    private function get_post_meta( $id ) {

        // Fetch all metadata
        $meta = get_post_meta( $id );

        // Bail if no metadata was returned
        if ( empty( $meta ) ) {
            return false;
        }

        // Define the metadata to ignore
        $ignored_keys = array(
            '_edit_lock',
            '_edit_last',
            '_page_generator_pro_settings',
            '_yoast_wpseo_content_score',
        );

        // Filter the ignored keys
        $ignored_keys = apply_filters( 'page_generator_pro_groups_get_meta_ignored_keys', $ignored_keys, $id );

        // Iterate through the metadata, removing items we don't want
        foreach ( $meta as $meta_key => $meta_value ) {
            // Remove ignored keys
            if ( in_array( $meta_key, $ignored_keys ) ) {
                unset( $meta[ $meta_key ] );
                continue;
            }

            // Fetch the single value
            $meta[ $meta_key ] = get_post_meta( $id, $meta_key, true );

            // Depending on the key, we might need to unserialize or decode the JSON
            switch ( $meta_key ) {
                case '_elementor_data':
                    // Decode JSON
                    if ( is_string( $meta[ $meta_key ] ) && ! empty( $meta[ $meta_key ] ) ) {
                        $meta[ $meta_key ] = json_decode( $meta[ $meta_key ], true );
                    }
                    if ( empty( $meta[ $meta_key ] ) ) {
                        $meta[ $meta_key ] = array();
                    }
                    break;
            }
        }

        // Return data
        return apply_filters( 'page_generator_pro_groups_get_meta', $meta );

    }

    /**
     * Returns an array of all Groups with their Settings
     *
     * @since   1.2.3
     *
     * @return  array   Groups
     */
    public function get_all() {

        // Groups
        $groups = new WP_Query( array(
            'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
            'post_status'   => 'any',
            'posts_per_page'=> -1,
        ) );

        if ( count( $groups->posts ) == 0 ) {
            return false;
        }

        // Build array
        $groups_arr = array();
        foreach ( $groups->posts as $group ) {
            // Get settings
            $groups_arr[ $group->ID ] = $this->get_settings( $group->ID );
        }

        // Return
        return apply_filters( 'page_generator_pro_groups_get_all', $groups_arr, $groups );

    }

    /**
     * Returns an array of all Group IDs with their names
     *
     * @since   1.2.3
     *
     * @return  array   Groups
     */
    public function get_all_ids_names() {

        // Groups
        $groups = new WP_Query( array(
            'post_type'             => Page_Generator_Pro_PostType::get_instance()->post_type_name,
            'post_status'           => 'any',
            'posts_per_page'        => -1,
            'update_post_term_cache'=> false,
            'update_post_meta_cache'=> false,
            'fields'                => 'ids',
        ) );

        if ( count( $groups->posts ) == 0 ) {
            return false;
        }

        // Build array
        $groups_arr = array();
        foreach ( $groups->posts as $group_id ) {
            $groups_arr[ $group_id ] = get_the_title( $group_id );
        }

        // Return
        return apply_filters( 'page_generator_pro_groups_get_all_ids_names', $groups_arr, $groups );

    }

    /**
     * Get the number of Groups
     *
     * @since   1.3.8
     *
     * @return  int             Number of Generated Pages / Posts / CPTs
     */
    public function get_count() {

        $posts = new WP_Query( array(
            'post_type'             => Page_Generator_Pro_PostType::get_instance()->post_type_name,
            'post_status'           => 'publish',
            'posts_per_page'        => 1,
            'update_post_term_cache'=> false,
            'update_post_meta_cache'=> false,
            'fields'                => 'ids',
        ) );

        return count( $posts->posts );

    }

    /**
     * Get the number of Pages / Posts / CPTs generated by the given Group ID
     *
     * @since   1.2.3
     *
     * @param   int     $id     Group ID
     * @return  int             Number of Generated Pages / Posts / CPTs
     */
    public function get_generated_count_by_id( $id ) {

        $posts = new WP_Query( array (
            'post_type'     => 'any',
            'post_status'   => 'publish',
            'posts_per_page'=> -1,
            'meta_query'    => array(
                array(
                    'key'   => '_page_generator_pro_group',
                    'value' => absint( $id ),
                ),
            ),
            'update_post_term_cache'=> false,
            'update_post_meta_cache'=> false,
            'fields'                => 'ids',
        ) );

        return count( $posts->posts );

    }

    /**
     * Called when a Group is saved.
     *
     * @since   1.2.3
     *
     * @param   int     $post_id
     */
    public function save_post( $post_id ) {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Bail if this isn't a Page Generator Pro Group that's being saved
        if ( get_post_type( $post_id ) != Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
            return;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return;
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], 'save_generate' ) ) {
            return;
        }

        // Call the main save function
        $this->save( $_POST[ $this->base->plugin->name ], $post_id, $_POST );

        // Check which submit action was given, as we may need to run a test or redirect to the generate screen now.
        $action = $this->get_action();
        if ( ! $action ) {
            return;
        }

        // Fetch group settings
        $settings = $this->get_settings( $post_id );
        if ( ! $settings ) {
            return new WP_Error( sprintf( __( 'Group ID %s does not exist!', $this->base->plugin->name ), $id ) );
        }

        // If here, we have a specific action to carry out
        $generate = Page_Generator_Pro_Generate::get_instance();
        switch ( $action ) {

            /**
             * Test
             */
            case 'test':
                $result = $generate->generate_content( $post_id, $settings['resumeIndex'], true );
                if ( is_wp_error( $result ) ) {
                    $notices['error'][] = $result->get_error_message();
                } else {
                    // Build success message
                    $message = __( 'Test Page Generated at ', $this->base->plugin->name ) . ': ' . '<a href="' . $result['url'] . '" target="_blank">' . $result['url'] . '</a>';
                    foreach ( $result['keywords_terms'] as $keyword => $term ) {
                        $message .= '<br />{' . $keyword . '}: ' . $term; 
                    }
                    $notices['success'][] = $message;
                }
                break;

            /**
             * Generate
             */
            case 'generate':
                wp_redirect( 'admin.php?page=' . $this->base->plugin->name . '-generate&id=' . $post_id . '&type=content' );
                die();
                break;

            /**
             * Delete Generated Content
             */
            case 'delete':
                $result = $generate->delete_content( $post_id );
                if ( is_wp_error( $result ) ) {
                    $notices['error'][] = $result->get_error_message();
                } else {
                    $notices['success'][] = __( 'Generated content deleted successfully.', $this->base->plugin->name );
                }
                break;

        }

        // Store success and/or error notices in a transient, which we'll load on the Post Edit redirect
        // and display, before clearing the transient.
        if ( isset( $notices ) ) {
            $user = wp_get_current_user();
            set_transient( 'page_generator_pro_groups_notices_' . $post_id . '_' . $user->ID, $notices, 15 );
        }

    }

    /**
     * When saving the Canonical URL field, revert Yoast SEO's sanitization, which strips curly braces
     *
     * @since   1.6.7
     *
     * @param   string  $clean          Sanitized Value
     * @param   string  $meta_value     Unsanitized Value
     * @param   array   $field_def      Meta Field Definition
     * @param   string  $meta_key       Meta Key
     * @return  string                  Value to save
     */
    public function wpseo_sanitize_post_meta__yoast_wpseo_canonical( $clean, $meta_value, $field_def, $meta_key ) {

        // Bail if no Post ID set
        if ( ! isset( $_POST['ID'] ) ) {
            return $clean;
        }

        // Get Post ID
        $group_id = absint( $_POST['ID'] );

        // Bail if not a Group
        if ( get_post_type( $group_id ) != Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
            return $clean;
        }

        // Return non-sanitized value
        return $meta_value;

    }

    /**
     * Determines which submit button was pressed on the Groups add/edit screen
     *
     * @since   1.2.3
     *
     * @return  string  Action
     */
    private function get_action() {

        if ( isset( $_POST['test'] ) ) {
            return 'test';
        }

        if ( isset( $_POST['generate'] ) ) {
            return 'generate';
        }

        if ( isset( $_POST['delete'] ) ) {
            return 'delete';
        }

        if ( isset( $_POST['save'] ) ) {
            return 'save';
        }

        // No action given
        return false;
  
    }

    /**
     * Adds or edits a record, based on the given settings array.
     *
     * @since   1.2.1
     * 
     * @param   array   $settings   Array of settings to save
     * @param   int     $id         Group ID
     * @param   array   $post_data  POST Data
     */
    public function save( $settings, $group_id, $post_data ) {

        // Merge with defaults, so keys are always set
        $settings = array_merge( $this->get_defaults(), $settings );

        // Clear out blank meta
        if ( isset( $settings['meta'] ) && is_array( $settings['meta'] ) && count( $settings['meta'] ) > 0 ) {
            foreach ( $settings['meta']['key'] as $index => $value ) {
                if ( empty( $value ) ) {
                    unset( $settings['meta']['key'][ $index ] );
                    unset( $settings['meta']['value'][ $index ] );
                }
            }
        }

        // Ensure some keys have a value, in case the user blanked out the values
        // This prevents errors later on when trying to generate content from a Group
        if ( empty( $settings['resumeIndex'] ) ) {
            $settings['resumeIndex'] = 0;
        }

        // Sanitize the Permalink setting
        if ( ! empty( $settings['permalink'] ) ) {
            $settings['permalink'] = preg_replace( "/[^a-z0-9-_{}:]+/i", "", str_replace( ' ', '-', $settings['permalink'] ) );
        }

        // Update Post Meta
        update_post_meta( $group_id, '_page_generator_pro_settings', $settings );

    }

    /**
     * Duplicates a group
     *
     * @since   1.2.3
     *
     * @param   int     $id     ID
     * @return  mixed           WP_Error | true
     */
    public function duplicate( $id ) {

        // Fetch group
        $post = get_post( $id );
        if ( ! $post ) {
            return new WP_Error( sprintf( __( 'Group ID %s does not exist!', $this->base->plugin->name ), $id ) );
        }

        // Fetch group settings
        $settings = $this->get_settings( $id );
        if ( ! $settings ) {
            return new WP_Error( sprintf( __( 'Group ID %s does not exist!', $this->base->plugin->name ), $id ) );
        }

        // Create new Post
        $duplicate_post_id = wp_insert_post( array(
            'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
            'post_title'    => $settings['title'] . __( ' - Copy', $this->base->plugin->name ),
            'post_content'  => $settings['content'],
            'post_status'   => $post->post_status,
        ) );

        // Bail if an error occured
        if ( is_wp_error( $duplicate_post_id ) ) {
            return $duplicate_post_id;
        }

        // Update Group Settings
        $this->save( $settings, $duplicate_post_id );

        // Duplicate Group Metadata
        $this->copy_post_meta( $duplicate_post_id, $settings['post_meta'] );

        // Redirect back to WP_List_Table screen
        // If we don't do this, subsequent actions would include the duplicate action and therefore keep duping Groups.
        wp_redirect( 'edit.php?post_type=' . Page_Generator_Pro_PostType::get_instance()->post_type_name );
        die();

    }

    /**
     * Copies all metadata from the source Group ID to the destination Group ID
     *
     * @since   1.4.4
     *
     * @param   int     $destination_group_id   Destination Group ID
     * @param   array   $meta_data              Group Metadata
     */
    private function copy_post_meta( $destination_group_id, $meta_data ) {

        // Bail if no data
        if ( empty( $meta_data ) ) {
            return;
        }

        // Iterate through the metadata, storing it in the new Group
        foreach ( $meta_data as $meta_key => $meta_value ) {
            update_post_meta( $destination_group_id, $meta_key, $meta_value );
        }
        
    }

    /**
     * Limit creating more than one Group via the WordPress Administration, by preventing
     * the 'Add New' functionality, and ensuring the user is always taken to the edit
     * screen of the single Group when they access the Post Type.
     *
     * @since   1.3.8
     */
    public function limit_admin() {

        global $pagenow;

        switch ( $pagenow ) {
            /**
             * Edit
             * WP_List_Table
             */
            case 'edit.php':
                // Bail if no Post Type is supplied
                if ( ! isset( $_REQUEST['post_type'] ) ) {
                    break;
                }

                // Fetch first group
                $groups = new WP_Query( array(
                    'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
                    'post_status'   => 'publish',
                    'posts_per_page'=> 1,
                ) );

                // Bail if no Groups exist, so the user can create one
                if ( count( $groups->posts ) == 0 ) {
                    break;
                }

                // Redirect to the Group's edit screen
                wp_safe_redirect( 'post.php?post=' . $groups->posts[0]->ID . '&action=edit' );
                die();

                break;

            /**
             * Add New
             */
            case 'post-new.php':
            case 'press-this.php':
                // Bail if we don't know the Post Type
                if ( ! isset( $_REQUEST['post_type'] ) ) {
                    break;
                }

                // Bail if we're not on our Group Post Type
                if ( $_REQUEST['post_type'] != Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
                    break;
                }

                // Fetch first group
                $groups = new WP_Query( array(
                    'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
                    'post_status'   => 'publish',
                    'posts_per_page'=> 1,
                ) );

                // Bail if no Groups exist, so the user can create one
                if ( count( $groups->posts ) == 0 ) {
                    break;
                }

                // Redirect to the Group's edit screen
                wp_safe_redirect( 'post.php?post=' . $groups->posts[0]->ID . '&action=edit' );
                die();
                
                break;
        }
            
    }

    /**
     * Limit creating more than one Group via XML-RPC
     *
     * @since   1.3.8
     *
     * @param   bool    $limit  Limit XML-RPC
     * @param   array   $post   Post Data
     * @return                  Limit XML-RPC
     */
    public function limit_xml_rpc( $limit, $post = array() ) {

        // Bail if we're not on an XMLRPC request
        if ( ! defined( 'XMLRPC_REQUEST' ) ||  XMLRPC_REQUEST != true ) {
            return $limit;
        }
        
        // Bail if no Post Type specified
        if ( ! isset( $post['post_type'] ) ) {
            return $limit;
        }
        if ( $post['post_type'] != Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
            return $limit;
        }

        // If here, we're trying to create a Group. Don't let this happen.
        return true;

    }
    
    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.2.1
     *
     * @return  object  Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}

// Load the class
$page_generator_pro_groups = Page_Generator_Pro_Groups::get_instance();