<?php
/**
 * Handles creating, editing, deleting and calling the generate routine
 * for the Generate Terms section of the Plugin.
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.6.1
 */
class Page_Generator_Pro_Groups_Terms {

    /**
     * Holds the class object.
     *
     * @since   1.6.1
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the base class object.
     *
     * @since   1.6.1
     *
     * @var     object
     */
    public $base;

    /**
     * Holds the common class object.
     *
     * @since   1.6.1
     *
     * @var     object
     */
    public $common;

    /**
     * Stores Keywords available to the Group
     *
     * @since   1.6.1
     *
     * @var     array
     */
    public $keywords;

    /**
     * Stores a Group's settings
     *
     * @since   1.6.1
     *
     * @var     array
     */
    public $settings;

    /**
     * Stores success and error messages
     *
     * @since   1.6.1
     *
     * @var     array
     */
    public $notices = array(
        'success'   => array(),
        'error'     => array(),
    );

    /**
     * Constructor
     *
     * @since   1.6.1
     */
    public function __construct() {

        // Process any notices that need to be displayed
        add_action( 'admin_notices', array( $this, 'admin_notices' ) );

        // WP_List_Table Columns
        add_filter( 'manage_edit-page-generator-tax_columns', array( $this, 'admin_columns' ) );
        add_filter( 'manage_page-generator-tax_custom_column', array( $this, 'admin_columns_output' ), 10, 3 );

        // Output the Permalink from the Term Settings, not the Term itself
        add_filter( 'editable_slug', array( $this, 'get_permalink_from_settings' ), 10, 2 );

        // Form Views
        add_action( 'page-generator-tax_term_edit_form_top', array( $this, 'output_taxonomy_start' ), 9, 2 );
        add_action( 'page-generator-tax_edit_form_fields', array( $this, 'output_taxonomy_fields' ), 10, 2 );
        add_action( 'page-generator-tax_edit_form', array( $this, 'output_taxonomy_end' ), 13, 2 );

        // Save Settings
        add_action( 'edit_term', array( $this, 'save_term' ), 10, 3 );

    }

    /**
     * Checks the transient to see if any admin notices need to be output now
     * i.e. if a Test or Delete call was made on this Group.
     *
     * @since   1.6.1
     */
    public function admin_notices() {

        // Get screen
        $screen = get_current_screen();

        // Bail if we're not editing a Taxonomy Term belonging to this Plugin
        if ( $screen->base != 'term' ) {
            return;
        }
        if ( $screen->taxonomy != Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name ) {
            return;
        }

        // Bail if there is no tag ID
        if ( ! isset( $_REQUEST['tag_ID'] ) ) {
            return;
        }

        // Get Term ID
        $term_id = absint( $_REQUEST['tag_ID'] );

        // Get current user
        $user = wp_get_current_user();
        
        // Check the transient for any notices
        $notices = get_transient( 'page_generator_pro_groups_notices_' . $term_id . '_' . $user->ID );
        if ( empty( $notices ) ) {
            return;
        }

        // If here, notice(s) exist
        // Store them in the class variable and delete the transient
        if ( isset( $notices['success'] ) ) {
            $this->notices['success'] = $notices['success'];
        }
        if ( isset( $notices['error'] ) ) {
            $this->notices['error'] = $notices['error'];
        }
        delete_transient( 'page_generator_pro_groups_notices_' . $term_id . '_' . $user->ID );

        // Output success notice(s)
        if ( is_array( $this->notices['success'] ) ) {
            foreach ( $this->notices['success'] as $message ) {
                ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php echo $message; ?></p>
                </div>
                <?php
            }
        }

        // Output error notice(s)
        if ( is_array( $this->notices['error'] ) ) {
            foreach ( $this->notices['error'] as $message ) {
                ?>
                <div class="notice notice-error is-dismissible">
                    <p><?php echo $message; ?></p>
                </div>
                <?php
            }
        }

    }

    /**
     * Adds columns to the Groups Terms within the WordPress Administration List Table
     * 
     * @since   1.6.1
     *
     * @param   array   $columns    Columns
     * @return  array               New Columns
     */
    public function admin_columns( $columns ) {

        // Remove columsn we don't want
        unset( $columns['posts'] );

        // Inject columns
        $columns['taxonomy']        = __( 'Taxonomy', $this->base->plugin->name );
        $columns['generated_count'] = __( 'No. Generated Items', $this->base->plugin->name );

        // Filter columns
        $columns = apply_filters( 'page_generator_pro_groups_terms_admin_columns', $columns );

        // Return
        return $columns;

    }


    /**
     * Manages the data to be displayed within a column on the Groups Taxonomy within 
     * the WordPress Administration List Table
     * 
     * @since   1.6.1
     *
     * @param   string  $content        Content
     * @param   string  $column_name    Column Name
     * @param   int     $term_id        Group ID
     */
    public function admin_columns_output( $content, $column_name, $term_id ) {

        switch ( $column_name ) {
            /**
             * Taxonomy
             */
            case 'taxonomy':
                // Get group settings
                $settings = $this->get_settings( $term_id );
                $content = $settings['taxonomy'];
                break;

            /**
             * Number of Generated Pages
             */
            case 'generated_count':
                // Get group settings
                $settings = $this->get_settings( $term_id );
                $content = $settings['generated_pages_count'];
                break;

            default:
                $content = apply_filters( 'page_generator_pro_groups_terms_admin_columns_output', $column_name, $term_id );
                break;
        }

        // Return
        return $content;

    }

    /**
     * When outputting the Slug on the Edit Term form, use the Slug from the settings,
     * and not the Term itself.
     *
     * The slug from the Term itself is sanitized, with non alpha-numeric characters removed,
     * including keywords.  This means the user does not see the 'true' value that we use
     * when defining the Term's permalink.
     *
     * @since   1.6.1
     *
     * @param   string      $slug   Slug
     * @param   WP_Term     $term   Term
     * @return  string              Slug
     */
    public function get_permalink_from_settings( $slug, $term ) {

        // Don't do anything if the Term's taxonomy is not our Plugin's
        if ( $term->taxonomy != Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name ) {
            return $slug;
        }

        // Get Settings
        $settings = $this->get_settings( $term->term_id );

        // Return Settings Permalink
        return $settings['permalink'];

    }

    /**
     * Defines a default settings structure when creating a new group
     *
     * @since   1.6.1
     *
     * @return  array   Group
     */
    public function get_defaults() {

        // Define defaults
        $defaults = array(
            'title'         => '',
            'permalink'     => '',
            'excerpt'       => '',
            'taxonomy'      => '',
            'method'        => 'all',
            'overwrite'     => 0,
            'numberOfPosts' => 0,
            'resumeIndex'   => 0,
            'numberOfPosts' => 0,
        );

        // Allow devs to filter defaults.
        $defaults = apply_filters( 'page_generator_pro_groups_terms_get_defaults', $defaults );

        // Return.
        return $defaults;
        
    }

    /**
     * Returns a Group's Settings by the given Group ID
     *
     * @since   1.6.1
     *
     * @param   int     $id     ID
     * @return  mixed           false | array
     */
    public function get_settings( $id ) {

        // Get settings
        $settings = get_term_meta( $id, '_page_generator_pro_settings', true );

        // If the result isn't an array, we're getting settings for a new Group, so just use the defaults
        if ( ! is_array( $settings ) ) {
            $settings = $this->get_defaults();
        } else {
            // Merge with defaults, so keys are always set
            $settings = array_merge( $this->get_defaults(), $settings );  
        }

        // Add the generated pages count
        $settings['generated_pages_count'] = $this->get_generated_count_by_id( $id, $settings['taxonomy'] );

        // Return settings
        return $settings;

    }

    /**
     * Get the number of Terms generated by the given Group ID
     *
     * @since   1.6.1
     *
     * @param   int     $id         Group ID
     * @param   string  $taxonomy   Taxonomy containing generated Terms
     * @return  int                 Number of Generated Terms
     */
    public function get_generated_count_by_id( $id, $taxonomy ) {

        $terms = new WP_Term_Query( array(
            'taxonomy'      => $taxonomy,
            'hide_empty'    => false,
            'meta_query'    => array(
                array(
                    'key'   => '_page_generator_pro_group',
                    'value' => absint( $id ),
                ),
            ),

            // For performance, just return the Post ID and don't update meta or term caches
            'fields'                => 'ids',
            'update_term_meta_cache'=> false,
        ) );

        if ( is_null( $terms->terms ) ) {
            return 0;
        }

        return count( $terms->terms );

    }

    /**
     * Outputs the HTML for Generating Terms, to provide a two column layout
     *
     * @since   1.6.1
     *
     * @param   WP_Term     $term       Term
     * @param   string      $taxonomy   Taxonomy
     */
    public function output_taxonomy_start( $term, $taxonomy ) {

        // Get instances
        $this->base   = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );
        $this->common = Page_Generator_Pro_Common::get_instance();

        // Get settings, as this is the first meta box to load
        $this->settings = $this->get_settings( $term->term_id );

        // Get options
        $methods = $this->common->get_methods();
        $overwrite_methods = $this->common->get_term_overwrite_methods();

        // Define labels
        $labels = array(
            'singular'  => __( 'Term', 'page-generator-pro' ),
            'plural'    => __( 'Terms', 'page-generator-pro' ),
        );

        // Output view
        require_once( $this->base->plugin->folder . '/views/admin/generate-taxonomy-wrap-start.php' );

    }

    /**
     * Adds additional fields to the Edit Term Form
     *
     * @since   1.6.1
     *
     * @param   WP_Term     $term       Term
     * @param   string      $taxonomy   Taxonomy
     */
    public function output_taxonomy_fields( $term, $taxonomy ) {

        // Get options
        $taxonomies = $this->common->get_taxonomies();

        // Output view
        require_once( $this->base->plugin->folder . '/views/admin/generate-taxonomy-fields.php' );

    }

    /**
     * Outputs HTML to close the Edit Taxonomy view.
     *
     * @since   1.6.1
     *
     * @param   WP_Term     $term       Term
     * @param   string      $taxonomy   Taxonomy
     */
    public function output_taxonomy_end( $term, $taxonomy ) {

        require_once( $this->base->plugin->folder . '/views/admin/generate-taxonomy-wrap-end.php' );

    }

    /**
     * Called when a Taxonomy Group is saved.
     *
     * @since   1.6.1
     *
     * @param   int     $term_id            Term ID
     * @param   int     $taxonomy_term_id   Taxonomy Term ID
     * @param   string  $taxonomy           Taxonomy
     */
    public function save_term( $term_id, $taxonomy_term_id, $taxonomy ) {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Bail if this isn't a Page Generator Taxonomy that's being saved
        if ( $taxonomy != Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name ) {
            return;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return;
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], 'save_generate' ) ) {
            return;
        }

        // Build the settings
        $settings = array_merge( array(
            'title'         => sanitize_text_field( $_POST['name'] ),
            'permalink'     => sanitize_text_field( $_POST['slug'] ),
            'excerpt'       => sanitize_text_field( $_POST['description'] ),
            'taxonomy'      => sanitize_text_field( $_POST['tax'] ),
        ), $_POST[ $this->base->plugin->name ] );

        // Call the main save function
        $this->save( $settings, $term_id );

        // Check which submit action was given, as we may need to run a test or redirect to the generate screen now.
        $action = $this->get_action();
        if ( ! $action ) {
            return;
        }

        // If here, we have a specific action to carry out
        $generate = Page_Generator_Pro_Generate::get_instance();
        switch ( $action ) {

            /**
             * Test
             */
            case 'test':
                $result = $generate->generate_term( $term_id, 0, true );
                
                if ( is_wp_error( $result ) ) {
                    $notices['error'][] = $result->get_error_message();
                } else {
                    // Build success message
                    $message = __( 'Test Term Generated at ', $this->base->plugin->name ) . ': ' . '<a href="' . $result['url'] . '" target="_blank">' . $result['url'] . '</a>';
                    foreach ( $result['keywords_terms'] as $keyword => $term ) {
                        $message .= '<br />{' . $keyword . '}: ' . $term; 
                    }
                    $notices['success'][] = $message;
                }
                break;

            /**
             * Generate
             */
            case 'generate':
                wp_redirect( 'admin.php?page=' . $this->base->plugin->name . '-generate&id=' . $term_id . '&type=term' );
                die();
                break;

            /**
             * Delete Generated
             */
            case 'delete':
                $result = $generate->delete_terms( $term_id );
                if ( is_wp_error( $result ) ) {
                    $notices['error'][] = $result->get_error_message();
                } else {
                    $notices['success'][] = __( 'Generated Terms deleted successfully.', $this->base->plugin->name );
                }
                break;

        }

        // Store success and/or error notices in a transient, which we'll load on the Term Edit redirect
        // and display, before clearing the transient.
        if ( isset( $notices ) ) {
            $user = wp_get_current_user();
            set_transient( 'page_generator_pro_groups_notices_' . $term_id . '_' . $user->ID, $notices, 15 );
        }

    }

    /**
     * Adds or edits a record, based on the given settings array.
     *
     * @since   1.6.1
     * 
     * @param   array  $settings    Array of settings to save
     * @param   int    $id          Group ID
     */
    public function save( $settings, $group_id ) {

        // Merge with defaults, so keys are always set
        $settings = array_merge( $this->get_defaults(), $settings );

        // Ensure some keys have a value, in case the user blanked out the values
        // This prevents errors later on when trying to generate content from a Group
        if ( empty( $settings['resumeIndex'] ) ) {
            $settings['resumeIndex'] = 0;
        }

        // Sanitize the Permalink setting
        if ( ! empty( $settings['permalink'] ) ) {
            $settings['permalink'] = preg_replace( "/[^a-z0-9-_{}]+/i", "", str_replace( ' ', '-', $settings['permalink'] ) );
        }

        // Update Term Meta
        update_term_meta( $group_id, '_page_generator_pro_settings', $settings );

    }

    /**
     * Determines which submit button was pressed on the Groups add/edit screen
     *
     * @since   1.6.1
     *
     * @return  string  Action
     */
    private function get_action() {

        if ( isset( $_POST['test'] ) ) {
            return 'test';
        }

        if ( isset( $_POST['generate'] ) ) {
            return 'generate';
        }

        if ( isset( $_POST['delete'] ) ) {
            return 'delete';
        }

        if ( isset( $_POST['save'] ) ) {
            return 'save';
        }

        // No action given
        return false;
  
    }

    /**
     * Duplicates a group
     *
     * @since   1.6.1
     *
     * @param   int     $id     ID
     * @return  mixed           WP_Error | true
     */
    public function duplicate( $id ) {

        // Fetch group
        $term = get_term( $id );
        if ( ! $term ) {
            return new WP_Error( sprintf( __( 'Group ID %s does not exist!', $this->base->plugin->name ), $id ) );
        }

        // Fetch group settings
        $settings = $this->get_settings( $id );
        if ( ! $settings ) {
            return new WP_Error( sprintf( __( 'Group ID %s does not exist!', $this->base->plugin->name ), $id ) );
        }

        // Create new Term
        $duplicate_term_id = wp_create_term( $settings['title'] . __( ' - Copy', $this->base->plugin->name ), $term->taxonomy, array(
            'slug'          => $settings['permalink'],
            'description'   => $settings['excerpt'],
            'parent'        => $settings['parent'],
        ) );

        // Bail if an error occured
        if ( is_wp_error( $duplicate_term_id ) ) {
            return $duplicate_term_id;
        }

        // Update Group Settings
        $this->save( $settings, $duplicate_term_id );

        // Redirect back to WP_List_Table screen
        // If we don't do this, subsequent actions would include the duplicate action and therefore keep duping Groups.
        wp_redirect( 'edit.php?post_type=' . Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name );
        die();

    }
    
    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.6.1
     *
     * @return  object  Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}

// Load the class
$page_generator_pro_groups_terms = Page_Generator_Pro_Groups_Terms::get_instance();