<?php
/**
 * geonames.org API Wrapper
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.5.1
 */
class Page_Generator_Pro_Geonames {

    /**
     * Holds the class object.
     *
     * @since   1.1.8
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the geonames.org username
     *
     * @since   1.1.8
     *
     * @var     string
     */
    private $username = 'wpzinc';

    /**
     * Holds the API endpoint
     *
     * @since   1.1.8
     *
     * @var     string
     */
    private $endpoint = 'http://api.geonames.org/';

    /**
     * Sets the username to the given Username, overriding the default.
     *
     * @since   1.5.2
     *
     * @param   string  $username   Username
     */
    public function set_username( $username ) {

        $this->username = $username;

    }

    /**
     * Returns an array of nearby locations within a radius, based on a given latitude and longitude.
     *
     * @since   1.5.1
     *
     * @param   float   $lat        Latitude
     * @param   float   $lng        Longitude
     * @param   int     $radius     Radius
     * @param   int     $limit      Max. Number of Results
     * @return  mixed               array | WP_Error
     */
    public function find_nearby_postal_codes( $lat, $lng, $radius, $limit = 500 ) {

        // Run query
        $results = $this->get( 'findNearbyPostalCodesJSON', array(
            'lat'       => $lat,
            'lng'       => $lng,
            'radius'    => ( $radius * 1.6 ),
            'maxRows'   => $limit,
        ) );

        // Bail if an error occured
        if ( is_wp_error( $results ) ) {
            return $results;
        }

        // If no results, bail
        if ( ! isset( $results->postalCodes ) || empty( $results->postalCodes ) || count( $results->postalCodes ) == 0 ) {
            return new WP_Error( 'page_generator_pro_geonames_get_nearby_zip_codes', __( 'No results were found.', 'page-generator-pro' ) );
        }

        // Parse results
        $terms = array();
        foreach ( $results->postalCodes as $result ) {
            $terms[] = array(
                'city'      => $result->placeName,
                'county'    => $result->adminName1,
                'zipcode'   => $result->postalCode,
            );
        }

        // Return array
        return $terms;

    }

    /**
     * Performs a GET request using wp_remote_get()
     *
     * @since   1.5.1
     *
     * @param   string  $action     Action
     * @param   array   $args       Arguments
     * @return  array   $data       Result Data
     */
    private function get( $action, $args ) {

        // Add username to args
        $args['username'] = $this->username;

        // Build URL
        $url = $this->endpoint . $action . '?' . http_build_query( $args );

        // Run query
        $response = wp_remote_get( $url );

        // Bail if an error occured
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        // Get body
        $data = wp_remote_retrieve_body( $response );
        $json = json_decode( $data );

        // Bail if a status is returned; this means something went wrong
        if ( isset( $json->status ) ) {
            return new WP_Error( 'page_generator_pro_geonames_get', $json->status->message );
        }

        // Return
        return $json;

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.5.1
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}