<?php
/**
 * Cities class
 *
 * Performs remote queries to build lists of nearby cities.
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.1.7
 */
class Page_Generator_Pro_Geo {

    /**
     * Holds the class object.
     *
     * @since   1.1.7
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the API Key
     *
     * @since   1.2.1
     *
     * @var     string
     */
    public $google_maps_geocode_api_key = 'AIzaSyDQJjWCtWqtYGE7cVu9zHrNjX_kweNSJm8';

    /**
     * Holds the Google Maps Embed API Key, which has no quota
     *
     * @since   1.5.1
     *
     * @var     string
     */
    public $google_maps_embed_api_key = 'AIzaSyCNTEOso0tZG6YMSJFoaJEY5Th1stEWrJI';

    /**
     * Primary SQL Table
     *
     * @since   1.5.8
     *
     * @var     string
     */
    public $table = 'page_generator_area_codes';
    
    /**
     * Primary SQL Table Primary Key
     *
     * @since   1.5.8
     *
     * @var     string
     */
    public $key = 'id';

    /**
     * Activation routines for this Model
     *
     * @since   1.5.8
     *
     * @global  $wpdb   WordPress DB Object
     */
    public function activate() {

        global $wpdb;

        // Enable error output if WP_DEBUG is enabled.
        $wpdb->show_errors = true;

        // Create database tables
        $wpdb->query( " CREATE TABLE IF NOT EXISTS " . $wpdb->prefix . $this->table . " (
                            `id` int(10) NOT NULL AUTO_INCREMENT,
                            `country` varchar(200) NOT NULL,
                            `country_code` int(3) NOT NULL,
                            `city` varchar(200) NOT NULL,
                            `area_code` varchar(200) NOT NULL, 
                            PRIMARY KEY `id` (`id`),
                            KEY `country` (`country`)
                        ) ENGINE=MyISAM 
                        DEFAULT CHARSET=" . $wpdb->charset . "
                        AUTO_INCREMENT=1" ); 

        // If the table has data, we've already inserted the data
        $count = $wpdb->get_var( "SELECT COUNT(*) FROM " . $wpdb->prefix . $this->table );
        if ( $count > 0 ) {
            return;
        }

        // Download CSV
        $data = file_get_contents( 'https://www.aggdata.com/download_sample.php?file=globalareacodes.csv' );

        // Iterate through rows, to build MySQL query
        $query = array();
        $rows = explode( "\n", $data );
        foreach ( $rows as $index => $row ) {
            // Skip column names
            if ( $index == 0 ) {
                continue;
            }

            // Explode into array
            $row = explode( ',', str_replace( '"', '', $row ) );

            // Add to query
            $query[] = '("' . ( isset( $row[0] ) ? $row[0] : '' ) . '", "' . ( isset( $row[1] ) ? $row[1] : '' ) . '", "' . ( isset( $row[2] ) ? $row[2] : '' ) . '", "' . ( isset( $row[3] ) ? $row[3] : '' ) . '")';
        }

        // Convert to MySQL query
        $query = 'INSERT INTO ' . $wpdb->prefix . $this->table . '(country, country_code, city, area_code) VALUES ' . implode( ',', $query );

        // Run query
        $wpdb->query( $query );

    }

    /**
     * Returns an array comprising of the latitude and longitude for a given city and country
     * using the Google Geocoding API
     *
     * @since   1.1.7
     *
     * @param   string  $city       City
     * @param   string  $country    Country
     * @return  mixed               Array | WP_Error
     */
    public function google_get_lat_lng( $city, $country = '' ) {

        // Format address
        if ( ! empty( $country ) ) {
            $address = $city . ',' . $country;   
        } else {
            $address = $city;
        }

        // Run query
        $response = wp_remote_get( 'https://maps.google.com/maps/api/geocode/json?address=' . urlencode( $address ) . '&sensor=false&key=' . $this->google_maps_geocode_api_key );

        // Bail if an error occured
        if ( is_wp_error( $response ) ) {
            // Try the OpenStreetMap API
            return $this->openstreetmap_get_lat_lng( $city, $country );
        }

        // Get body
        $data = wp_remote_retrieve_body( $response );
        $json = json_decode( $data );

        // Check status
        if ( isset( $json->status ) && $json->status != 'OK' ) {
            // Try the OpenStreetMap API
            return $this->openstreetmap_get_lat_lng( $city, $country );
        }

        // Get lat/lng
        $lat = (float) $json->results[0]->geometry->location->lat;
        $lng = (float) $json->results[0]->geometry->location->lng;

        // Return
        return array(
            'lat' => $lat,
            'lng' => $lng,
        );

    }

    /**
     * Returns an array comprising of the latitude and longitude for a given city and country
     * using the OpenStreetMap API
     *
     * @since   1.1.7
     *
     * @param   string  $city       City
     * @param   string  $country    Country
     * @return  mixed               Array | WP_Error
     */
    public function openstreetmap_get_lat_lng( $city, $country = '' ) {

        // Format address
        if ( ! empty( $country ) ) {
            $address = $city . ',' . $country;   
        } else {
            $address = $city;
        }

        // Run query
        $response = wp_remote_get( 'http://nominatim.openstreetmap.org/search?q=' . urlencode( $address ) . '&format=json' );

        // Bail if an error occured
        if ( is_wp_error( $response ) ) {
            return $response;
        }

        // Get body
        $data = wp_remote_retrieve_body( $response );
        $json = json_decode( $data );

        // Check some results were found
        if ( ! is_array( $json ) || count( $json ) == 0 ) {
            return new WP_Error( 'page_generator_pro_geo_openstreetmap_get_lat_lng_error', sprintf( __( 'Could not get latitude and longitude for %s', 'page-generator-pro' ), $address ) );
        }

        // Get lat/lng from the first result
        $lat = (float) $json[0]->lat;
        $lng = (float) $json[0]->lon;

        // Return
        return array(
            'lat' => $lat,
            'lng' => $lng,
        );

    }

    /**
     * Returns an array of nearby cities and towns for the given latitude, longitude and radius.
     *
     * Each return result contains the city and county, which we then add
     * phone data to, to produce a 'full' result.
     *
     * @since   1.1.7
     *
     * @param   float   $lat        Latitude
     * @param   float   $lng        Longitude
     * @param   int     $radius     Radius (in miles)
     * @return  mixed               Array | WP_Error
     */
    public function get_nearby_cities( $lat, $lng, $radius ) {

        // Limit the radius to the permitted maximum of 100
        // If it's greater, we will get fewer results from the API, for some reason
        if ( $radius > 100 ) {
            $radius = 100;
        }

        // Use cURL to fetch data. wp_remote_get returns a 403, most likely because the UA isn't set correctly.
        $ch = curl_init();
        curl_setopt( $ch, CURLOPT_URL, 'http://www.geoplugin.net/extras/nearby.gp?lat=' . $lat . '&long=' . $lng . '&limit=100&radius=' . absint( $radius ) . '&format=json' );
        curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
        curl_setopt( $ch, CURLOPT_USERAGENT, 'geoPlugin PHP Class v1.0' );
        $response = curl_exec( $ch );
        $http_code = curl_getinfo( $ch, CURLINFO_HTTP_CODE );
        curl_close( $ch );

        // If the HTTP response code isn't 200, something went wrong
        if ( $http_code != 200 ) {
            return new WP_Error( 'page_generator_pro_geo_get_nearby_cities_no_results', sprintf( __( 'GeoPlugin returned HTTP Status Code %s. Please try again.', 'page-generator-pro' ), $http_code ) );
        }

        // Decode JSON
        $json = json_decode( $response );
       
        // Check cities were found
        if ( ! is_array( $json ) || count( $json ) == 0 ) {
            return new WP_Error( 'page_generator_pro_geo_get_nearby_cities_no_results', __( 'No nearby cities found. Try increasing the radius.', 'page-generator-pro' ) );
        }

        // Build array
        $cities = array();
        foreach ( $json as $city ) {
            $cities[] = array(
                'city'                          => $city->geoplugin_place,
                'county'                        => ( isset( $city->geoplugin_region ) ? $city->geoplugin_region : '' ),
            );
        }

        // Return cities
        return $cities;

    }

    /**
     * Returns an array of nearby zip codes for the given latitude, longitude and radius.
     *
     * Each return result contains the zip code, city and county, which we then add
     * phone data to, to produce a 'full' result.
     *
     * @since   1.5.1
     *
     * @param   float   $lat            Latitude
     * @param   float   $lng            Longitude
     * @param   int     $radius         Radius (in miles)
     * @param   string  $username       Geonames Username
     * @return  mixed                   Array | WP_Error
     */
    public function get_nearby_zipcodes( $lat, $lng, $radius, $username = '' ) {

        // Limit the radius to the permitted maximum of 18 miles (~ 30km)
        if ( $radius > 18 ) {
            $radius = 18;
        }

        // Get instance
        $instance = Page_Generator_Pro_Geonames::get_instance();

        // If a username has been specified, use it now
        if ( isset( $username ) ) {
            $instance->set_username( $username );
        }

        // Get data
        return $instance->find_nearby_postal_codes( $lat, $lng, $radius );

    }

    /**
     * Returns an array of supported countries for generating phone area codes
     *
     * @since   1.5.9
     *
     * @return  array   Countries
     */
    public function get_phone_area_code_countries() {

        global $wpdb;
       
        // Get records
        $query = "SELECT DISTINCT country FROM " . $wpdb->prefix . $this->table; 
        $results = $wpdb->get_results( $query, ARRAY_A );

        // Check a record was found     
        if ( ! $results ) {
            return false;
        }             
        if ( count( $results ) == 0 ) {
            return false;
        }

        // Build array
        $countries = array();
        foreach ( $results as $result ) {
            // Skip blank results
            if ( empty( $result['country'] ) ) {
                continue;
            }

            // Add to countries array
            $countries[ $result['country'] ] = $result['country'];
        }

        // Build unique array
        $countries = array_unique( $countries );

        // Return filtered results
        return apply_filters( 'page_generator_pro_geo_get_phone_area_code_countries', $countries );

    }

    /**
     * Returns area dialling codes for the given country
     *
     * @since   1.5.9
     *
     * @param   string  $country    Country
     * @return  array               Phone Data
     */
    public function get_phone_area_codes( $country ) {

        global $wpdb;
       
        // Get records
        $query = $wpdb->prepare("   SELECT area_code, city, country_code
                                    FROM " . $wpdb->prefix . $this->table . "
                                    WHERE country = '%s'",
                                    $country ); 
        $results = $wpdb->get_results( $query, ARRAY_A );

        // Check a record was found     
        if ( ! $results ) {
            return false;
        }             
        if ( count( $results ) == 0 ) {
            return false;
        }

        // Build array
        $area_codes = array();
        foreach ( $results as $result ) {
            // Skip blank results
            if ( empty( $result['area_code'] ) ) {
                continue;
            }

            // Add to area codes array
            $area_codes[ $result['area_code'] ] = $result;
        }

        // Return filtered results
        return apply_filters( 'page_generator_pro_geo_get_phone_area_codes', $area_codes );

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.7
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}