<?php
/**
 * Generate class. Handles generating Pages, Posts, Custom Post Types
 * and Taxonomy Terms.
 * 
 * @package Page_Generator_Pro
 * @author  Tim Carr
 * @version 1.0.0
 */
class Page_Generator_Pro_Generate {

    /**
     * Holds the class object.
     *
     * @since   1.1.3
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the array of found keywords across all settings.
     *
     * @since   1.2.0
     *
     * @var     array
     */
    public $required_keywords = array();

    /**
     * Holds the array of found keywords that require a fixed term across all settings.
     *
     * @since   1.2.0
     *
     * @var     array
     */
    public $required_keywords_fixed = array();

    /**
     * Holds the array of keywords to replace e.g. {city}
     *
     * @since   1.3.1
     *
     * @var     array
     */
    public $searches = array();

    /**
     * Holds the array of keyword values to replace e.g. Birmingham
     *
     * @since   1.3.1
     *
     * @var     array
     */
    public $replacements = array();

    /**
     * Calculates the maximum number of items that will be generated based
     * on the settings.
     *
     * @since   1.1.5
     *
     * @param   array   $settings   Group Settings (either a Content or Term Group)
     * @return  mixed               WP_Error | integer
     */
    public function get_max_number_of_pages( $settings ) {

        // Get instances
        $common_instance = Page_Generator_Pro_Common::get_instance();
        $groups_instance = Page_Generator_Pro_Groups::get_instance();
        
        // Get an array of required keywords that need replacing with data
        $required_keywords = $this->find_keywords_in_settings( $settings );

        // Bail if no keywords were found
        if ( count( $required_keywords ) == 0 ) {
            return 0;
        }

        // Get the terms for each required keyword
        $keywords = $this->get_keywords_terms( $required_keywords );

        // Bail if no keywords were found
        if ( empty( $keywords ) ) {
            return 0;
        }

        // Depending on the generation method chosen, for each keyword, define the term
        // that will replace it.
        switch ( $settings['method'] ) {

            /**
             * All
             * Random
             * - Generates all possible term combinations across keywords
             */
            case 'all':
            case 'random':
                $total = 1;
                foreach ( $keywords as $keyword => $terms ) {
                    $total = ( $total * count( $terms ) );
                }

                return $total;
                break;

            /**
             * Sequential
             * - Generates term combinations across keywords matched by index
             */
            case 'sequential':
                $total = 0;
                foreach ( $keywords as $keyword => $terms ) {
                    if ( count( $terms ) > 0 && ( count( $terms ) < $total || $total == 0 ) ) {
                        $total = count( $terms );
                    }
                }

                return $total;
                break;

        }

    }

    /**
     * Deprecated function for generating Posts, Pages and Custom Post Types
     * from a given Group.
     *
     * Use generate_content() instead.
     *
     * @since   1.0.0
     *
     * @param   int     $group_id   Group ID
     * @param   int     $index      Keyword Index
     * @param   bool    $test_mode  Test Mode
     * @return  mixed               WP_Error | array
     */
    public function generate( $group_id, $index = 0, $test_mode = false ) {

        return $this->generate_content( $group_id, $index, $test_mode );

    }
    
    /**
     * Main function to generate a Page, Post, Custom Post Type or Taxonomy Term
     *
     * @since   1.6.1
     *
     * @param   int     $group_id   Group ID
     * @param   int     $index      Keyword Index
     * @param   bool    $test_mode  Test Mode
     * @return  mixed               WP_Error | URL
     */
    public function generate_content( $group_id, $index = 0, $test_mode = false ) {

        // Get instances
        $common_instance    = Page_Generator_Pro_Common::get_instance();
        $groups_instance    = Page_Generator_Pro_Groups::get_instance();
        $spintax            = Page_Generator_Pro_Spintax::get_instance();
        
        // Get group settings
        $settings = $groups_instance->get_settings( $group_id );
        if ( ! $settings ) {
            return new WP_Error( 'group_error', sprintf( __( 'Group ID %s could not be found.', 'page-generator-pro' ), $group_id ) );
        }

        // Get an array of required keywords that need replacing with data
        $required_keywords = $this->find_keywords_in_settings( $settings );
        if ( count( $required_keywords ) == 0 ) {
            return new WP_Error( 'keyword_error', __( 'No keywords were specified in the Group.', 'page-generator-pro' ) );
            die();
        }

        // Get the terms for each required keywords
        $keywords = $this->get_keywords_terms( $required_keywords );

        // Bail if no keywords were found
        if ( count( $keywords ) == 0 ) {
            return new WP_Error( 'keyword_error', __( 'Keywords were specified in the Group, but no keywords exist in either the Keywords section of the Plugin or as a Taxonomy.', 'page-generator-pro' ) );
            die();
        }

        // Depending on the generation method chosen, for each keyword, define the term
        // that will replace it.
        switch ( $settings['method'] ) {

            /**
             * All
             * - Generates all possible term combinations across keywords
             */
            case 'all':
                // If we're on PHP 5.5+, use our Cartesian Product class, which implements a Generator
                // to allow iteration of data without needing to build an array in memory.
                // See: http://php.net/manual/en/language.generators.overview.php
                if ( version_compare( phpversion(), '5.5.0', '>=' ) ) {
                    // Use PHP 5.5+ Generator
                    $combinations = $this->generate_all_combinations( $keywords );

                    // If the current index exceeds the total number of combinations, we've exhausted all
                    // options and don't want to generate any more Pages (otherwise we end up with duplicates)
                    if ( $index > ( $combinations->count() - 1 ) ) {
                        // If the combinations count is a negative number, we exceeded the floating point for an integer
                        // Tell the user to upgrade PHP and/or reduce the number of keyword terms
                        if ( $combinations->count() < 0 ) {
                            $message = __( 'The total possible number of unique keyword term combinations exceeds the maximum number value that can be stored by your version of PHP.  Please consider upgrading to a 64 bit PHP 7.0+ build and/or reducing the number of keyword terms that you are using.', 'page-generator-pro' );
                        } else {
                            $message = __( 'All possible keyword term combinations have been generated. Generating more Pages/Posts would result in duplicate content.', 'page-generator-pro' );
                        }

                        return new WP_Error( 'page_generator_pro_generate_content_keywords_exhausted', $message );
                        die();
                    }

                    // Iterate through the combinations until we reach the one matching the index
                    // @TODO Can we optimize this?
                    foreach ( $combinations as $c_index => $combination ) {
                        // Skip if not the index we want
                        if ( $c_index != $index ) {
                            continue;
                        }

                        // Define the keyword => term key/value pairs to use based on the current index
                        $keywords_terms = $combination;
                        break;
                    }
                } else {
                    // Use older method, which will hit memory errors
                    $combinations = $this->generate_all_array_combinations( $keywords );  

                    // If the current index exceeds the total number of combinations, we've exhausted all
                    // options and don't want to generate any more Pages (otherwise we end up with duplicates)
                    if ( $index > ( count( $combinations ) - 1 ) ) {
                        return new WP_Error( 'keywords_exhausted', __( 'All possible keyword term combinations have been generated. Generating more Pages/Posts would result in duplicate content.', 'page-generator-pro' ) );
                        die();
                    }

                    // Define the keyword => term key/value pairs to use based on the current index
                    $keywords_terms = $combinations[ $index ];
                    break;
                }
                break;

            /**
             * Sequential
             * - Generates term combinations across keywords matched by index
             */
            case 'sequential':
                $keywords_terms = array();
                foreach ( $keywords as $keyword => $terms ) {
                    // Use modulo to get the term index for this keyword
                    $term_index = ( $index % count( $terms ) );   

                    // Build the keyword => term key/value pairs
                    $keywords_terms[ $keyword ] = $terms[ $term_index ];
                }
                break;

            /**
             * Random
             * - Gets a random term for each keyword
             */
            case 'random':
                $keywords_terms = array();
                foreach ( $keywords as $keyword => $terms ) {
                    $term_index = rand( 0, ( count( $terms ) - 1 ) );  

                    // Build the keyword => term key/value pairs
                    $keywords_terms[ $keyword ] = $terms[ $term_index ];
                }
                break;

        }

        // Rotate Author
        if ( isset( $settings['rotateAuthors'] ) ) {
            $authors = $common_instance->get_authors();
            $userIndex = ( $index % count( $authors ) );
        }

        // Iterate through each keyword and term key/value pair
        foreach ( $keywords_terms as $keyword => $term ) {

            // Define the search and replace queries
            // We have multiple queries as we're looking for:
            // - keyword: {keyword}
            // - keyword with transformation {keyword:uppercase_all}
            $term = trim( html_entity_decode( $term ) );
            $this->searches = array(
                '{' . $keyword . '}',                                   // Keyword Term
                '{' . $keyword . ':uppercase_all}',                     // Keyword Term, uppercase
                '{' . $keyword . ':lowercase_all}',                     // Keyword Term, lowercase
                '{' . $keyword . ':uppercase_first_character}',         // Keyword Term, first char uppercase
                '{' . $keyword . ':uppercase_first_character_words}',   // Keyword Term, first char of each word uppercase
                '{' . $keyword . ':url}',                               // Keyword Term, as a URL / permalink friendly string
            );
            $this->replacements = array(
                $term,
                strtoupper( $term ),
                strtolower( $term ),
                ucfirst( $term ),
                ucwords( $term ),
                sanitize_title( $term ),
            );

            // Add any {keyword:nth} searches and replacements now
            if ( count( $this->required_keywords_fixed ) > 0 && isset( $this->required_keywords_fixed[ $keyword ] ) && count( $this->required_keywords_fixed[ $keyword ] ) > 0 ) {
                foreach ( $this->required_keywords_fixed[ $keyword ] as $keyword_term_index ) {
                    // If the nth child is out of the keyword term array's bounds, ignore it
                    if ( ! isset( $keywords[ $keyword ][ $keyword_term_index - 1 ] ) ) {
                        continue;
                    }

                    $this->searches[] = '{' . $keyword . ':' . $keyword_term_index . '}';
                    $this->replacements[] = $keywords[ $keyword ][ $keyword_term_index - 1 ];
                }
            }

            // Go through each of the group's settings, replacing $search with $replacement 
            foreach ( $settings as $key => $value ) {
                // Depending on the setting key, process the search and replace
                switch ( $key ) {
                    /**
                     * Taxonomies
                     */
                    case 'tax':
                        // If the submitted taxonomies are an array, iterate through each one
                        // This allows hierarchical taxonomies to have keyword replacements carried out on nested arrays e.g.
                        // $settings[tax][category][0]
                        if ( is_array( $settings[ $key ] ) ) {
                            foreach ( $settings[ $key ] as $taxonomy => $terms ) {
                                // Hierarchical based taxonomy - first key may contain new tax terms w/ keywords that need replacing now
                                if ( is_array( $terms ) && isset( $terms[0] ) ) {
                                    $settings[ $key ][ $taxonomy ][0] = str_ireplace( $this->searches, $this->replacements, $settings[ $key ][ $taxonomy ][0] );
                                }

                                // Tag based taxonomy
                                if ( ! is_array( $terms ) ) {
                                    $settings[ $key ][ $taxonomy ] = str_ireplace( $this->searches, $this->replacements, $settings[ $key ][ $taxonomy ] );   
                                }
                            }
                        }
                        break;

                    /**
                     * Default
                     * - Will also cover keyword search / replace for Page Builders data
                     */
                    default:
                        // Don't do anything if there's no data
                        if ( empty( $settings[ $key ] ) ) {
                            break;
                        }
 
                        // If the settings key's value is an array, walk through it recursively to search/replace
                        // Otherwise do a standard search/replace on the string
                        if ( is_array( $settings[ $key ] ) ) {
                            // Array
                            array_walk_recursive( $settings[ $key ], array( $this, 'replace_keywords_in_array' ) );
                        } elseif( is_object( $settings[ $key ] ) ) {
                            // Object
                            array_walk_recursive( $settings[ $key ], array( $this, 'replace_keywords_in_array' ) );
                        } else {
                            // Keyword search/replace
                            $settings[ $key ] = str_ireplace( $this->searches, $this->replacements, $settings[ $key ] );   
                        }
                        break;

                }
            } 
        }

        // Spin all settings
        foreach ( $settings as $key => $value ) {
            // Skip if value is not a string
            if ( ! is_string( $value ) ) {
                continue;
            }

            // Spin content
            $settings[ $key ] = $spintax->process( $settings[ $key ] );              
        }

        // Remove all shortcode processors, so we don't process any shortcodes. This ensures page builders, galleries etc
        // will work as their shortcodes will be processed when the generated page is viewed.
        remove_all_shortcodes();

        // Add Page Generator Pro's shortcodes, so they're processed now.
        Page_Generator_Pro_Shortcode::get_instance()->add_shortcodes( true );
        
        // Execute shortcodes in content, so actual HTML is output instead of shortcodes for this plugin's shortcodes
        $content = do_shortcode( $settings['content'] );

        // Determine the Post Parent
        $post_parent = ( ( isset( $settings['pageParent'] ) && isset( $settings['pageParent'][ $settings['type'] ] ) && ! empty( $settings['pageParent'][ $settings['type'] ] ) ) ? $settings['pageParent'][ $settings['type'] ] : 0 );
        if ( ! is_numeric( $post_parent ) ) {
            // Find the Post ID based on the given name
            $parent = get_page_by_path( $post_parent, OBJECT, $settings['type'] );

            if ( ! $parent ) {
                $post_parent = 0;
            } else {
                $post_parent = $parent->ID;
            }
        }

        // Build Post args
        $post_args = array(
            'post_type'     => $settings['type'],
            'post_title'    => $spintax->process( $settings['title'] ),
            'post_content'  => $content,
            'post_excerpt'  => $spintax->process( $settings['excerpt'] ),
            'post_status'   => ( $test_mode ? 'draft' : $settings['status'] ),
            'post_author'   => ( ( isset( $settings['rotateAuthors'] ) && $settings['rotateAuthors'] == 1 ) ? $authors[ $userIndex ]->ID : $settings['author'] ), // ID
            'comment_status'=> ( ( isset( $settings['comments'] ) && $settings['comments'] == 1 ) ? 'open' : 'closed' ),
            'ping_status'   => ( ( isset( $settings['trackbacks'] ) && $settings['trackbacks'] == 1 ) ? 'open' : 'closed' ),
            'post_parent'   => $post_parent,
        );

        // Define Post Name
        // If no Permalink exists, use the Post Title
        if ( ! empty( $settings['permalink'] ) ) {
            $post_args['post_name'] = sanitize_title( $settings['permalink'] );
        } else {
            $post_args['post_name'] = sanitize_title( $post_args['post_title'] );
        }

        // Define the Post Date
        switch ( $settings['date_option'] ) {

            /**
            * Now
            */
            case 'now':
                if ( $settings['status'] == 'future' ) {
                    // Increment the current date by the schedule hours and unit
                    $post_args['post_date'] = date_i18n( 'Y-m-d H:i:s', strtotime( '+' . ( $settings['schedule'] * ( $index + 1 ) ) . ' ' . $settings['scheduleUnit'] ) );
                } else {
                    $post_args['post_date'] = date_i18n( 'Y-m-d H:i:s' );
                }
                break;

            /**
            * Specific Date
            */
            case 'specific':
                if ( $settings['status'] == 'future' ) {
                    // Increment the specific date by the schedule hours and unit
                    $post_args['post_date'] = date_i18n( 'Y-m-d H:i:s', strtotime( $settings['date_specific'] ) . ' +' . ( $settings['schedule'] * ( $index + 1 ) ) . ' ' . $settings['scheduleUnit'] );
                } else {
                    $post_args['post_date'] = $settings['date_specific'];
                }
                break;

            /**
            * Random
            */
            case 'random':
                $min = strtotime( $settings['date_min'] );
                $max = strtotime( $settings['date_max'] );
                $post_args['post_date'] = date_i18n( 'Y-m-d H:i:s', rand( $min, $max ) );
                break;

        }

        // Allow filtering
        $post_args = apply_filters( 'page_generator_pro_generate_post_args', $post_args, $settings );

        // If overwrite is enabled, attempt to find an existing Post generated by this Group
        // with the same Permalink
        switch ( $settings['overwrite'] ) {

            /**
             * Overwrite
             */
            case 'overwrite':
            case 'overwrite_preseve_date':
                // Try to find existing post
                $existing_post = new WP_Query( array(
                    'post_type'     => $post_args['post_type'],
                    'post_status'   => array( 'publish', 'pending', 'draft', 'future', 'private' ),
                    'post_name__in' => array( $post_args['post_name'] ),
                    'meta_query'    => array(
                        array(
                            'key'   => '_page_generator_pro_group',
                            'value' => $group_id,
                        ),
                    ),

                    // For performance, just return the Post ID and don't update meta or term caches
                    'fields'                => 'ids',
                    'cache_results'         => false,
                    'update_post_meta_cache'=> false,
                    'update_post_term_cache'=> false,
                ) );

                // If a Post was found, update it
                if ( count( $existing_post->posts ) > 0 ) {
                    // Define the Post ID to update
                    $post_args['ID']    = $existing_post->posts[0];

                    // Don't update the date if the settings require it to be preserved
                    if ( $settings['overwrite'] == 'overwrite_preseve_date' ) {
                        unset( $post_args['post_date'] );
                    }

                    // Update Page, Post or CPT
                    $post_id = wp_update_post( $post_args, true ); 
                } else {
                    // Create Page, Post or CPT
                    $post_id = wp_insert_post( $post_args, true ); 
                }
                break;

            /**
             * Don't Overwrite
             */
            default:
                // Create Page, Post or CPT
                $post_id = wp_insert_post( $post_args, true );
                break;

        }

        // Check Post creation / update worked
        if ( is_wp_error( $post_id ) ) {
            // UTF-8 encode the Title, Excerpt and Content
            $post_args['post_title'] = utf8_encode( $post_args['post_title'] );
            $post_args['post_excerpt'] = utf8_encode( $post_args['post_excerpt'] );
            $post_args['post_content'] = utf8_encode( $post_args['post_content'] );
            
            // Try again
            if ( count( $existing_post->posts ) > 0 ) {
                // Update Page, Post or CPT
                $post_id            = wp_update_post( $post_args, true ); 
            } else {
                // Create Page, Post or CPT
                $post_id            = wp_insert_post( $post_args, true ); 
            }
        }

        // If Post creation / update still didn't work, bail
        if ( is_wp_error( $post_id ) ) {
            $post_id->add_data( $post_args, $post_id->get_error_code() );
            return $post_id;
        }

        // Store this Generation ID in the Post's meta, so we can edit/delete the generated Post(s) in the future
        update_post_meta( $post_id, '_page_generator_pro_group', $group_id );

        // Post / Page Template
        if ( ! empty( $settings['pageTemplate'][ $settings['type'] ] ) ) {
            update_post_meta( $post_id, '_wp_page_template', $settings['pageTemplate'][ $settings['type'] ] );
        }

        // Featured Image
        if ( ! empty( $settings['featured_image_source'] ) ) {
            switch ( $settings['featured_image_source'] ) {
                /**
                 * Media Library ID
                 */
                case 'id':
                    $image_id = $settings['featured_image'];
                    break;

                /**
                 * Image URL
                 */
                case 'url':
                    $image_id = Page_Generator_Pro_Import::get_instance()->import_remote_image( $settings['featured_image'], $post_id );
                    break;

            }

            // If an image ID is specified, and it's not an error, set it now
            if ( isset( $image_id ) && ! is_wp_error( $image_id ) ) {
                update_post_meta( $post_id, '_thumbnail_id', $image_id );

                // If an ALT tag was specified, set that against the Media Library image ID now
                if ( ! empty( $settings['featured_image_alt'] ) ) {
                    update_post_meta( $image_id, '_wp_attachment_image_alt', $spintax->process( $settings['featured_image_alt'] ) );
                }
            }
        }

        // Custom Fields
        if ( isset( $settings['meta'] ) ) {
            foreach ( $settings['meta']['key'] as $meta_index => $meta_key ) {
                $meta_value = $spintax->process( $settings['meta']['value'][ $meta_index ] );
                update_post_meta( $post_id, $meta_key, $meta_value );
            }
        }

        // Post Meta
        // This will copy e.g. ACF, Page Builder data etc.
        if ( isset( $settings['post_meta'] ) ) {
            foreach ( $settings['post_meta'] as $meta_key => $meta_value ) {
                // Some meta keys need to be handled differently
                switch ( $meta_key ) {
                    case '_elementor_data':
                        // Encode with slashes, just how Elementor does
                        $meta_value = wp_slash( wp_json_encode( $meta_value ) );
                        
                        // Store using update_metadata, just how Elementor does
                        update_metadata( 'post', $post_id, $meta_key, $meta_value );
                        break;

                    case '_wp_page_template':
                        // Skip this, as it will overwrite the template that we define in the group settings
                        break;

                    default:
                        update_post_meta( $post_id, $meta_key, $meta_value );
                        break;
                }
            }
        }
        
        // Taxonomies
        // Get taxonomies for this Post Type
        // @TODO Move spintax to keyword search/replace routine above
        if ( isset( $settings['tax'] ) ) {
            $taxonomies = $common_instance->get_post_type_taxonomies( $settings['type'] );
            $ignored_taxonomies = $common_instance->get_excluded_taxonomies();
            if ( is_array( $taxonomies ) && count( $taxonomies ) > 0 ) {
                // Iterate through taxonomies
                foreach ( $taxonomies as $taxonomy ) {
                    // Skip ignored taxonomies
                    if ( in_array( $taxonomy->name, $ignored_taxonomies ) ) {
                        continue;
                    }

                    // Clear vars from the last iteration
                    unset( $terms );

                    // Check if hierarchal or tag based
                    switch ( $taxonomy->hierarchical ) { 

                        case true:
                            // Category based taxonomy
                            if ( isset( $settings['tax'][ $taxonomy->name ] ) ) {
                                $terms = array();
                                foreach ( $settings['tax'][ $taxonomy->name ] as $taxID => $enabled ) {
                                    // If tax ID is zero, the value will be a string of new taxonomy terms we need to create
                                    if ( $taxID == 0 ) {
                                        // String
                                        $terms_string = $spintax->process( $enabled );

                                        // Convert to array
                                        $terms_arr = explode( ',', $terms_string );

                                        // Add each term to the taxonomy
                                        foreach ( $terms_arr as $new_term ) {
                                            // Check if this named term already exists in the taxonomy
                                            $result = term_exists( $new_term, $taxonomy->name );
                                            if ( $result !== 0 && $result !== null ) {
                                                $terms[] = (int) $result['term_id'];
                                                continue;
                                            }

                                            // Term does not exist in the taxonomy - create it
                                            $result = wp_insert_term( $new_term, $taxonomy->name );
                                            
                                            // Skip if something went wrong
                                            if ( is_wp_error( $result ) ) {
                                                continue;
                                            }
                                            
                                            // Add to term IDs
                                            $terms[] = (int) $result['term_id'];
                                        }
                                        continue;
                                    }

                                    $terms[] = (int) $taxID;
                                }

                            }  
                            break;

                        case false:
                            // Tag based taxonomy
                            $terms = $spintax->process( $settings['tax'][ $taxonomy->name ] );
                            break;

                    }

                    // Set terms if they exist
                    if ( isset( $terms ) ) {
                        $result = wp_set_post_terms( $post_id, $terms, $taxonomy->name, false );
                    }

                }
            }
        }

        // Get URL of Page/Post/CPT just generated
        $url = get_bloginfo( 'url' ) . '?page_id=' . $post_id . '&preview=true';

        // Request that the user review the plugin. Notification displayed later,
        // can be called multiple times and won't re-display the notification if dismissed.
        if ( ! $test_mode ) {
            if ( class_exists( 'Page_Generator' ) ) {
                Page_Generator::get_instance()->dashboard->request_review();    
            } else {
                Page_Generator_Pro::get_instance()->dashboard->request_review();
            }
        }

        // Return the URL and keyword / term replacements used
        return array(
            'url'           => $url,
            'keywords_terms'=> $keywords_terms,
        );

    }

    /**
     * Main function to generate a Page, Post, Custom Post Type or Taxonomy Term
     *
     * @since   1.0.0
     *
     * @param   int     $group_id   Group ID
     * @param   int     $index      Keyword Index
     * @param   bool    $test_mode  Test Mode
     * @return  mixed               WP_Error | URL
     */
    public function generate_term( $group_id, $index, $test_mode = false ) {

        // Get instances
        $common_instance    = Page_Generator_Pro_Common::get_instance();
        $groups_instance    = Page_Generator_Pro_Groups_Terms::get_instance();
        $spintax            = Page_Generator_Pro_Spintax::get_instance();
        
        // Get group settings
        $settings = $groups_instance->get_settings( $group_id );
        if ( ! $settings ) {
            return new WP_Error( 'group_error', sprintf( __( 'Group ID %s could not be found.', 'page-generator-pro' ), $group_id ) );
        }

        // Get an array of required keywords that need replacing with data
        $required_keywords = $this->find_keywords_in_settings( $settings );
        if ( count( $required_keywords ) == 0 ) {
            return new WP_Error( 'keyword_error', __( 'No keywords were specified in the title, content or excerpt.', 'page-generator-pro' ) );
        }

        // Get the terms for each required keyword
        $keywords = $this->get_keywords_terms( $required_keywords );

        // Bail if no keywords were found
        if ( count( $keywords ) == 0 ) {
            return new WP_Error( 'keyword_error', __( 'Keywords were specified in the Group, but no keywords exist in either the Keywords section of the Plugin or as a Taxonomy.', 'page-generator-pro' ) );
            die();
        }

        // Depending on the generation method chosen, for each keyword, define the term
        // that will replace it.
        switch ( $settings['method'] ) {

            /**
             * All
             * - Generates all possible term combinations across keywords
             */
            case 'all':
                // If we're on PHP 5.5+, use our Cartesian Product class, which implements a Generator
                // to allow iteration of data without needing to build an array in memory.
                // See: http://php.net/manual/en/language.generators.overview.php
                if ( version_compare( phpversion(), '5.5.0', '>=' ) ) {
                    // Use PHP 5.5+ Generator
                    $combinations = $this->generate_all_combinations( $keywords );

                    // If the current index exceeds the total number of combinations, we've exhausted all
                    // options and don't want to generate any more Pages (otherwise we end up with duplicates)
                    if ( $index > ( $combinations->count() - 1 ) ) {
                        // If the combinations count is a negative number, we exceeded the floating point for an integer
                        // Tell the user to upgrade PHP and/or reduce the number of keyword terms
                        if ( $combinations->count() < 0 ) {
                            $message = __( 'The total possible number of unique keyword term combinations exceeds the maximum number value that can be stored by your version of PHP.  Please consider upgrading to a 64 bit PHP 7.0+ build and/or reducing the number of keyword terms that you are using.', 'page-generator-pro' );
                        } else {
                            $message = __( 'All possible keyword term combinations have been generated. Generating more Pages/Posts would result in duplicate content.', 'page-generator-pro' );
                        }

                        return new WP_Error( 'page_generator_pro_generate_terms_keywords_exhausted', $message );
                        die();
                    }

                    // Iterate through the combinations until we reach the one matching the index
                    // @TODO Can we optimize this?
                    foreach ( $combinations as $c_index => $combination ) {
                        // Skip if not the index we want
                        if ( $c_index != $index ) {
                            continue;
                        }

                        // Define the keyword => term key/value pairs to use based on the current index
                        $keywords_terms = $combination;
                        break;
                    }
                } else {
                    // Use older method, which will hit memory errors
                    $combinations = $this->generate_all_array_combinations( $keywords );

                    // If the current index exceeds the total number of combinations, we've exhausted all
                    // options and don't want to generate any more Pages (otherwise we end up with duplicates)
                    if ( $index > ( count( $combinations ) - 1 ) ) {
                        return new WP_Error( 'keywords_exhausted', __( 'All possible keyword term combinations have been generated. Generating more Pages/Posts would result in duplicate content.', 'page-generator-pro' ) );
                    }

                    // Define the keyword => term key/value pairs to use based on the current index
                    $keywords_terms = $combinations[ $index ];
                    break;
                }
                break;

            /**
             * Sequential
             * - Generates term combinations across keywords matched by index
             */
            case 'sequential':
                $keywords_terms = array();
                foreach ( $keywords as $keyword => $terms ) {
                    // Use modulo to get the term index for this keyword
                    $term_index = ( $index % count( $terms ) );   

                    // Build the keyword => term key/value pairs
                    $keywords_terms[ $keyword ] = $terms[ $term_index ];
                }
                break;

            /**
             * Random
             * - Gets a random term for each keyword
             */
            case 'random':
                $keywords_terms = array();
                foreach ( $keywords as $keyword => $terms ) {
                    $term_index = rand( 0, ( count( $terms ) - 1 ) );  

                    // Build the keyword => term key/value pairs
                    $keywords_terms[ $keyword ] = $terms[ $term_index ];
                }
                break;

        }

        // Iterate through each keyword and term key/value pair
        foreach ( $keywords_terms as $keyword => $term ) {

            // Define the search and replace queries
            // We have multiple queries as we're looking for:
            // - keyword: {keyword}
            // - keyword with transformation {keyword:uppercase_all}
            $term = trim( html_entity_decode( $term ) );
            $this->searches = array(
                '{' . $keyword . '}',                                   // Keyword Term
                '{' . $keyword . ':uppercase_all}',                     // Keyword Term, uppercase
                '{' . $keyword . ':lowercase_all}',                     // Keyword Term, lowercase
                '{' . $keyword . ':uppercase_first_character}',         // Keyword Term, first char uppercase
                '{' . $keyword . ':uppercase_first_character_words}',   // Keyword Term, first char of each word uppercase
                '{' . $keyword . ':url}',                               // Keyword Term, as a URL / permalink friendly string
            );
            $this->replacements = array(
                $term,
                strtoupper( $term ),
                strtolower( $term ),
                ucfirst( $term ),
                ucwords( $term ),
                sanitize_title( $term ),
            );
                
            // Go through each of the group's settings, replacing $search with $replacement 
            foreach ( $settings as $key => $value ) {
                // Depending on the setting key, process the search and replace
                switch ( $key ) {
                    /**
                     * Taxonomies
                     */
                    case 'tax':
                        // If the submitted taxonomies are an array, iterate through each one
                        // This allows hierarchical taxonomies to have keyword replacements carried out on nested arrays e.g.
                        // $settings[tax][category][0]
                        if ( is_array( $settings[ $key ] ) ) {
                            foreach ( $settings[ $key ] as $taxonomy => $terms ) {
                                // Hierarchical based taxonomy - first key may contain new tax terms w/ keywords that need replacing now
                                if ( is_array( $terms ) && isset( $terms[0] ) ) {
                                    $settings[ $key ][ $taxonomy ][0] = str_ireplace( $this->searches, $this->replacements, $settings[ $key ][ $taxonomy ][0] );
                                }

                                // Tag based taxonomy
                                if ( ! is_array( $terms ) ) {
                                    $settings[ $key ][ $taxonomy ] = str_ireplace( $this->searches, $this->replacements, $settings[ $key ][ $taxonomy ] );   
                                }
                            }
                        }
                        break;

                    /**
                     * Default
                     * - Will also cover keyword search / replace for Page Builders data
                     */
                    default:
                        // Don't do anything if there's no data
                        if ( empty( $settings[ $key ] ) ) {
                            break;
                        }

                        // If the settings key's value is an array, walk through it recursively to search/replace
                        // Otherwise do a standard search/replace on the string
                        if ( is_array( $settings[ $key ] ) ) {
                            // Array
                            array_walk_recursive( $settings[ $key ], array( $this, 'replace_keywords_in_array' ) );
                        } elseif( is_object( $settings[ $key ] ) ) {
                            // Object
                            array_walk_recursive( $settings[ $key ], array( $this, 'replace_keywords_in_array' ) );
                        } else {
                            // Keyword search/replace
                            $settings[ $key ] = str_ireplace( $this->searches, $this->replacements, $settings[ $key ] );   
                        }
                        break;

                }
            }
        }

        // Spin all settings
        foreach ( $settings as $key => $value ) {
            // Skip if value is not a string
            if ( ! is_string( $value ) ) {
                continue;
            }

            // Spin content
            $settings[ $key ] = $spintax->process( $settings[ $key ] );              
        }

        // Build Term args
        $term_args = array(
            'description'   => $settings['excerpt'],
        );

        // Define Slug
        // If no Permalink exists, use the Title
        if ( ! empty( $settings['permalink'] ) ) {
            $term_args['slug'] = sanitize_title( $settings['permalink'] );
        }

        // Allow filtering
        $term_args = apply_filters( 'page_generator_pro_generate_term_args', $term_args, $settings );

        // Check if the Term we're about to generate already exists
        $existing_terms = new WP_Term_Query( array(
            'taxonomy'      => array( $settings['taxonomy'] ),
            'name'          => array( $settings['title'] ),
            'hide_empty'    => false,
            
            // For performance, just return the Post ID and don't update meta or term caches
            'fields'                => 'ids',
            'update_term_meta_cache'=> false,
        ) );

        // If overwrite is enabled, attempt to find an existing Term with the same Permalink
        switch ( $settings['overwrite'] ) {

            /**
             * Overwrite
             */
            case 'overwrite':
                // If a Post was found, update it
                if ( ! is_null( $existing_terms->terms ) && count( $existing_terms->terms ) > 0 ) {
                    // Update Term
                    $term = wp_update_term( $existing_terms->terms[0], $settings['taxonomy'], $term_args );
                } else {
                    // Create Term
                    $term = wp_insert_term( $settings['title'], $settings['taxonomy'], $term_args ); 
                }
                break;

            /**
             * Don't Overwrite
             */
            default:
                // If a Post was found, update it
                if ( ! is_null( $existing_terms->terms ) && count( $existing_terms->terms ) > 0 ) {
                    // Return existing URL
                    return array(
                        'url'           => get_term_link( $existing_terms->terms[0], $settings['taxonomy'] ),
                        'keywords_terms'=> $keywords_terms,
                    );
                } else {
                    // Create Term
                    $term = wp_insert_term( $settings['title'], $settings['taxonomy'], $term_args ); 
                }
                break;

        }
        
        // Check Term creation / update worked
        if ( is_wp_error( $term ) ) {
            $term->add_data( $term_args, $term->get_error_code() );
            return $term;
        }

        // Store this Generation ID in the Post's meta, so we can edit/delete the generated Post(s) in the future
        update_term_meta( $term['term_id'], '_page_generator_pro_group', $group_id );

        // Get URL of Term just generated
        $url = get_term_link( $term['term_id'], $settings['taxonomy'] );

        // Request that the user review the plugin. Notification displayed later,
        // can be called multiple times and won't re-display the notification if dismissed.
        if ( ! $test_mode ) {
            Page_Generator_Pro::get_instance()->dashboard->request_review();
        }

        // Return the URL and keyword / term replacements used
        return array(
            'url'           => $url,
            'keywords_terms'=> $keywords_terms,
        );

    }

    /**
     * Returns an array comprising of keywords, with each keyword having a replacement value, based
     * on the index requested.
     *
     * 
     * Returns an array comprising of all possible value combinations, for the given keywords and terms
     *
     * @since   1.5.1
     *
     * @param   array   $input  Multidimensional array of Keyword Names (keys) => Terms (values)
     * @return  array           Single dimensional array, zero indexed, of keyword names (keys) => term (value)
     */
    private function generate_all_array_combinations( $input ) {

        // Setup vars
        $input  = array_filter( $input );
        $result = array( array() );

        // Iterate through each keyword
        foreach ( $input as $keyword => $terms ) {
            $append = array();

            // Iterate through master array of results
            foreach ( $result as $product ) {
                // Iterate through this keyword's terms
                foreach( $terms as $term ) {
                    $product[ $keyword ] = $term;
                    $append[] = $product;
                }
            }

            // Append the list of Terms to the master array of results
            $result = $append;
        }

        return $result;

    }

    /**
     * A faster method for fetching all keyword combinations for PHP 5.5+
     *
     * @since   1.5.1
     *
     * @param   array   $input  Multidimensional array of Keyword Names (keys) => Terms (values)
     * @return  \Generator      Generator
     */
    private function generate_all_combinations( $input ) {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator' ) ? Page_Generator::get_instance() : Page_Generator_Pro::get_instance() );

        // Load class
        require_once( $this->base->plugin->folder . '/includes/admin/cartesian-product.php' );

        // Return
        return new Page_Generator_Pro_Cartesian_Product( $input );

    }

    /**
     * Recursively goes through the settings array, finding any {keywords}
     * specified, to build up an array of keywords we need to fetch.
     *
     * @since   1.0.0
     *
     * @param   array   $settings   Settings
     * @return  array               Found Keywords
     */
    private function find_keywords_in_settings( $settings ) {

        // Get all keywords
        $keywords = Page_Generator_Pro_Keywords::get_instance()->get_all( 'keyword', 'ASC', -1 );

        // Recursively walk through all settings to find all keywords
        array_walk_recursive( $settings, array( $this, 'find_keywords_in_string' ) );

        // Return the required keywords object
        return $this->required_keywords;

    }

    /**
     * For the given array of keywords, only returns keywords with terms, where keywords
     * have terms.
     *
     * @since   1.6.5
     *
     * @param   array   $required_keywords  Required Keywords
     * @return  array                       Keywords with Terms
     */
    private function get_keywords_terms( $required_keywords ) {

        // Get keywords instance
        $keywords_instance = Page_Generator_Pro_Keywords::get_instance();

        // Define blank array for keywords with terms
        $keywords = array();

        foreach ( $required_keywords as $key => $keyword ) {

            // Assume this keyword has no terms
            $terms = false;

            // Get terms for this keyword
            // If this keyword starts with 'taxonomy_', try to fetch the terms for the Taxonomy
            if ( strpos( $keyword, 'taxonomy_') !== false && strpos( $keyword, 'taxonomy_' ) == 0 ) {
                $result = get_terms( array(
                    'taxonomy'              => str_replace( 'taxonomy_', '', $keyword ),
                    'hide_empty'            => false,
                    'fields'                => 'names',
                    'update_term_meta_cache'=> false,
                ) );

                if ( is_array( $result ) && count( $result ) > 0 ) {
                    $terms = $result;
                }
            } else {
                $result = $keywords_instance->get_by( 'keyword', $keyword );
                
                if ( is_array( $result ) && isset( $result['dataArr'] ) && count( $result['dataArr'] ) > 0 ) {
                    $terms = $result['dataArr'];
                }
            }

            // Skip this keyword if no terms were found
            if ( ! $terms || count( $terms ) == 0 ) {
                continue;
            }

            // Add this keyword and its terms to the keywords array
            $keywords[ $keyword ] = $terms;

        }

        return $keywords;

    }

    /**
     * Performs a search on the given string to find any {keywords}
     *
     * @since 1.2.0
     *
     * @param   string  $content    Array Value (string to search)
     * @param   string  $key        Array Key
     */
    private function find_keywords_in_string( $content, $key ) {

        // Define array to store found keywords in
        $required_keywords = array();
        $required_keywords_fixed = array();

        // If $content is an object, iterate this call
        if ( is_object( $content ) ) {
            return array_walk_recursive( $content, array( $this, 'find_keywords_in_string' ) );
        }

        // Get keywords and spins
        preg_match_all( "|{(.+?)}|", $content, $matches );

        // Continue if no matches found
        if ( ! is_array( $matches ) ) {
            return;
        }
        if ( count( $matches[1] ) == 0 ) {
            return;
        }

        // Iterate through matches
        foreach ( $matches[1] as $m_key => $keyword ) {
            // Ignore spins
            if ( strpos( $keyword, "|" ) !== false ) {
                continue;
            }

            // If a keyword is within spintax at the start of the string (e.g. {{service}|{service2}} ),
            // we get an additional leading curly brace for some reason.  Remove it
            $keyword = str_replace( '{', '', $keyword );
            $keyword = str_replace( '}', '', $keyword );

            // If there's a transformation flag applied to the keyword, remove it
            if ( strpos( $keyword, ':' ) !== false ) {
                list( $keyword, $transformation ) = explode( ':', $keyword );                
            }

            // Lowercase keyword, to avoid duplicates e.g. {City} and {city}
            $keyword = strtolower( $keyword );

            // If this keyword is not in our required_keywords array, add it
            if ( ! in_array( $keyword, $required_keywords ) ) {
                $required_keywords[ $keyword ] = $keyword;
            }

            // If the transformation is a number, add this keyword to the required keywords fixed array
            if ( ! is_numeric( $transformation ) ) {
                continue;
            }
            if ( ! is_array( $required_keywords_fixed[ $keyword ] ) ) {
                $required_keywords_fixed[ $keyword ] = array(
                    $transformation,
                );
            } else {
                $required_keywords_fixed[ $keyword ][] = $transformation;
            }
        }

        // Add the found keywords to the class array
        $this->required_keywords = array_merge( $this->required_keywords, $required_keywords );
        $this->required_keywords_fixed = array_merge( $this->required_keywords_fixed, $required_keywords_fixed );

    }

    /**
     * array_walk_recursive callback, which finds $this->searches, replacing with
     * $this->replacements in $item
     *
     * @since   1.3.1
     *
     * @param   mixed   $item   Item (array, object, string)
     * @param   string  $key    Key
     */
    private function replace_keywords_in_array( &$item, $key ) {

        // If $item is an object, iterate this call
        if ( is_object( $item ) ) {
            array_walk_recursive( $item, array( $this, 'replace_keywords_in_array' ) );
        } else {
            $item = str_ireplace( $this->searches, $this->replacements, $item );
        }

        // If $item is a string, spintax it
        if ( is_string( $item ) ) {
            $item = Page_Generator_Pro_Spintax::get_instance()->process( $item );
        }

    }

    /**
     * Main function to delete previously generated Contents
     * for the given Group ID
     *
     * @since   1.2.3
     *
     * @param   int     $group_id   Group ID
     * @return  mixed               WP_Error | Success
     */
    public function delete_content( $group_id ) {

        // Get all Posts
        $posts = new WP_Query( array (
            'post_type'     => 'any',
            'post_status'   => 'publish',
            'posts_per_page'=> -1,
            'meta_query'    => array(
                array(
                    'key'   => '_page_generator_pro_group',
                    'value' => absint( $group_id ),
                ),
            ),
            'fields'        => 'ids',
        ) );

        // If no Posts found, return false, as there's nothing to delete
        if ( count( $posts->posts ) == 0 ) {
            return new WP_Error( __( 'No content has been generated by this group, so there is no content to delete.', 'page-generator-pro' ) );
        }

        // Delete Posts by their IDs
        foreach ( $posts->posts as $post_id ) {
            $result = wp_trash_post( $post_id );
            if ( ! $result ) {
                return new WP_Error( __( 'Unable to delete generated content with ID = ' . $post_id, 'page-generator-pro' ) );
            }
        }

        // Done
        return true;

    }

    /**
     * Main function to delete previously generated Terms
     * for the given Group ID
     *
     * @since   1.6.1
     *
     * @param   int     $group_id   Group ID
     * @return  mixed               WP_Error | Success
     */
    public function delete_terms( $group_id ) {

        // Get Settings
        $groups_instance    = Page_Generator_Pro_Groups_Terms::get_instance();
        $settings           = $groups_instance->get_settings( $group_id );

        // Get all Terms
        $terms = new WP_Term_Query( array(
            'taxonomy'      => $settings['taxonomy'],
            'meta_query'    => array(
                array(
                    'key'   => '_page_generator_pro_group',
                    'value' => absint( $group_id ),
                ),
            ),
            'hide_empty'    => false,
            
            // For performance, just return the Post ID and don't update meta or term caches
            'fields'                => 'ids',
            'update_term_meta_cache'=> false,
        ) );

        // If no Terms found, return false, as there's nothing to delete
        if ( count( $terms->terms ) == 0 ) {
            return new WP_Error( __( 'No Terms have been generated by this group, so there are no Terms to delete.', 'page-generator-pro' ) );
        }

        // Delete Terms by their IDs
        foreach ( $terms->terms as $term_id ) {
            $result = wp_delete_term( $term_id, $settings['taxonomy'] );
            if ( ! $result ) {
                return new WP_Error( __( 'Unable to delete generated Term with ID = ' . $term_id, 'page-generator-pro' ) );
            }
        }

        // Done
        return true;

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.3
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}