<?php
/**
 * Common class
 * 
 * @package Page Generator Pro
 * @author  Tim Carr
 * @version 1.0.0
 */
class Page_Generator_Pro_Common {

    /**
     * Holds the class object.
     *
     * @since   1.1.3
     *
     * @var     object
     */
    public static $instance;

    /**
     * Helper method to retrieve public Post Types
     *
     * @since   1.1.3
     *
     * @return  array   Public Post Types
     */
    public function get_post_types() {

        // Get public Post Types
        $types = get_post_types( array(
            'public' => true,
        ), 'objects' );

        // Filter out excluded post types
        $excluded_types = $this->get_excluded_post_types();
        if ( is_array( $excluded_types ) ) {
            foreach ( $excluded_types as $excluded_type ) {
                unset( $types[ $excluded_type ] );
            }
        }

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_post_types', $types );

    }

    /**
     * Helper method to retrieve hierarchical public Post Types
     *
     * @since   1.2.1
     *
     * @return  array   Public Post Types
     */
    public function get_hierarchical_post_types() {

        // Get public hierarchical Post Types
        $types = get_post_types( array(
            'public'        => true,
            'hierarchical'  => true,
        ), 'objects' );

        // Filter out excluded post types
        $excluded_types = $this->get_excluded_post_types();
        if ( is_array( $excluded_types ) ) {
            foreach ( $excluded_types as $excluded_type ) {
                unset( $types[ $excluded_type ] );
            }
        }

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_hierarchical_post_types', $types );

    }

    /**
     * Helper method to retrieve excluded Post Types
     *
     * @since   1.1.3
     *
     * @return  array   Excluded Post Types
     */
    public function get_excluded_post_types() {

        // Get excluded Post Types
        $types = array(
            'attachment',
            'revision',
            'nav_menu_item',
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_excluded_post_types', $types );

    }

    /**
     * Returns any available Templates for each Post Type
     *
     * @since   1.5.8
     *
     * @return  array   Post Types and Templates
     */
    public function get_post_types_templates() {

        // Get Post Types
        $post_types = $this->get_post_types();

        // Bail if no Post Types
        if ( empty( $post_types ) ) {
            return false;
        }

        // Build templates
        $templates = array();
        foreach ( $post_types as $post_type ) {
            // Skip if this Post Type doesn't have any templates
            $post_type_templates = get_page_templates( null, $post_type->name );
            if ( empty( $post_type_templates ) ) {
                continue;
            }
            
            $templates[ $post_type->name ] = $post_type_templates;
        }

        return apply_filters( 'page_generator_pro_common_get_post_type_templates', $templates );

    }

    /**
     * Helper method to retrieve all Taxonomies
     *
     * @since   1.1.3
     *
     * @param   string  $post_type  Post Type
     * @return  array               Taxonomies
     */
    public function get_taxonomies() {

        // Get all taxonomies
        $taxonomies = get_taxonomies();

        // Get information for each taxonomy
        foreach ( $taxonomies as $index => $taxonomy ) {
            // Ignore our own taxonomy
            if ( $taxonomy == Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name ) {
                unset( $taxonomies[ $index ] );
                continue;
            }
            
            $taxonomies[ $index ] = get_taxonomy( $taxonomy );
        }

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_taxonomies', $taxonomies );

    }

    /**
     * Helper method to retrieve Taxonomies for the given Post Type
     *
     * @since   1.1.3
     *
     * @param   string  $post_type  Post Type
     * @return  array               Taxonomies
     */
    public function get_post_type_taxonomies( $post_type = '' ) {

        // Get Post Type Taxonomies
        $taxonomies = get_object_taxonomies( $post_type, 'objects' );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_post_type_taxonomies', $taxonomies, $post_type );

    }

    /**
     * Helper method to retrieve excluded Taxonomies
     *
     * @since   1.1.3
     *
     * @return  array   Taxonomies
     */
    public function get_excluded_taxonomies() {

        // Get excluded Taxonomies
        $taxonomies = array(
            'nav_menu',
            'link_category',
            'post_format',
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_excluded_taxonomies', $taxonomies );

    }

    /**
     * Helper method to retrieve authors
     *
     * @since   1.1.3
     *
     * @return  array   Authors
     */
    public function get_authors() {

        // Get authors
        $authors = get_users( array(
             'orderby' => 'nicename',
        ) );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_authors', $authors );
        
    }

    /**
     * Helper method to retrieve post statuses
     *
     * @since   1.1.3
     *
     * @return  array   Post Statuses
     */
    public function get_post_statuses() {

        // Get statuses
        $statuses = array(
            'draft'     => __( 'Draft', 'page-generator-pro' ),
            'future'    => __( 'Scheduled', 'page-generator-pro' ),
            'private'   => __( 'Private', 'page-generator-pro' ),
            'publish'   => __( 'Publish', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_post_statuses', $statuses );
        
    }

    /**
     * Helper method to retrieve post date options
     *
     * @since   1.1.6
     *
     * @return  array   Date Options
     */
    public function get_date_options() {

        // Get date options
        $date_options = array(
            'now'       => __( 'Now', 'page-generator-pro' ),
            'specific'  => __( 'Specify Date', 'page-generator-pro' ),
            'random'    => __( 'Random Date', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_date_options', $date_options );

    }

    /**
     * Helper method to retrieve post schedule units
     *
     * @since   1.1.3
     *
     * @return  array   Post Schedule Units
     */
    public function get_schedule_units() {

        // Get units
        $units = array(
            'minutes'   => __( 'Minutes', 'page-generator-pro' ),
            'hours'     => __( 'Hours', 'page-generator-pro' ),
            'days'      => __( 'Days', 'page-generator-pro' ),
            'weeks'     => __( 'Weeks', 'page-generator-pro' ),
            'months'    => __( 'Months', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_schedule_units', $units );
        
    }

    /**
     * Helper method to retrieve Post/Page generation methods
     *
     * @since   1.1.5
     *
     * @return  array   Generation Methods
     */
    public function get_methods() {

        // Get methods
        $methods = array(
            'all'        => __( 'All', 'page-generator-pro' ),
            'sequential' => __( 'Sequential', 'page-generator-pro' ),
            'random'     => __( 'Random', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_methods', $methods );

    }

    /**
     * Helper method to retrieve available overwrite generation methods
     *
     * @since   1.5.8
     *
     * @return  array   Overwrite Methods
     */
    public function get_overwrite_methods() {

        // Get methods
        $methods = array(
            0                       => __( 'No', 'page-generator-pro' ),
            'overwrite'             => __( 'Yes', 'page-generator-pro' ),
            'overwrite_preseve_date'=> __( 'Yes, preseving original Publish date', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_overwrite_methods', $methods );

    }

    /**
     * Helper method to retrieve available overwrite generation methods
     * for Terms
     *
     * @since   1.6.1
     *
     * @return  array   Overwrite Methods
     */
    public function get_term_overwrite_methods() {

        // Get overwrite methods
        $methods = $this->get_overwrite_methods();

        // Remove non-applicable methods
        unset( $methods['overwrite_preseve_date'] );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_term_overwrite_methods', $methods );

    }

    /**
     * Helper method to retrieve Featured Image source options
     *
     * @since   1.1.7
     *
     * @return  array   Featured Image Sources
     */
    public function get_featured_image_sources() {

        // Get sources
        $sources = array(
            ''        => __( 'No Feat. Image', 'page-generator-pro' ),
            'id'      => __( 'Media Library Image', 'page-generator-pro' ),
            'url'     => __( 'Image URL', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_featured_image_sources', $sources );

    }

    /**
     * Helper method to retrieve Order By options
     *
     * @since   1.7.2
     *
     * @return  array   Order By Options
     */
    public function get_order_by_options() {

        // Get order by options
        $order_by = array(
            'none'          => __( 'No Order', 'page-generator-pro' ),
            'ID'            => __( 'Post ID', 'page-generator-pro' ),
            'author'        => __( 'Author', 'page-generator-pro' ),
            'title'         => __( 'Title', 'page-generator-pro' ),
            'name'          => __( 'Name', 'page-generator-pro' ),
            'type'          => __( 'Post Type', 'page-generator-pro' ),
            'date'          => __( 'Published Date', 'page-generator-pro' ),
            'modified'      => __( 'Modified Date', 'page-generator-pro' ),
            'rand'          => __( 'Random', 'page-generator-pro' ),
            'comment_count' => __( 'Number of Comments', 'page-generator-pro' ),
            'relevance'     => __( 'Relevance', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_order_by_options', $order_by );

    }

    /**
     * Helper method to retrieve Order options
     *
     * @since   1.7.2
     *
     * @return  array   Order Options
     */
    public function get_order_options() {

        // Get order options
        $order = array(
            'asc'          => __( 'Ascending (A-Z)', 'page-generator-pro' ),
            'desc'         => __( 'Descending (Z-A)', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_order_options', $order );

    }

    /**
     * Helper method to retrieve country codes and names
     *
     * @since   1.1.7
     *
     * @return  array   Countries
     */
    public function get_countries() {

        // Get countries
        $countries = array (
            'AF' => 'Afghanistan',
            'AX' => 'Aland Islands',
            'AL' => 'Albania',
            'DZ' => 'Algeria',
            'AS' => 'American Samoa',
            'AD' => 'Andorra',
            'AO' => 'Angola',
            'AI' => 'Anguilla',
            'AQ' => 'Antarctica',
            'AG' => 'Antigua And Barbuda',
            'AR' => 'Argentina',
            'AM' => 'Armenia',
            'AW' => 'Aruba',
            'AU' => 'Australia',
            'AT' => 'Austria',
            'AZ' => 'Azerbaijan',
            'BS' => 'Bahamas',
            'BH' => 'Bahrain',
            'BD' => 'Bangladesh',
            'BB' => 'Barbados',
            'BY' => 'Belarus',
            'BE' => 'Belgium',
            'BZ' => 'Belize',
            'BJ' => 'Benin',
            'BM' => 'Bermuda',
            'BT' => 'Bhutan',
            'BO' => 'Bolivia',
            'BA' => 'Bosnia And Herzegovina',
            'BW' => 'Botswana',
            'BV' => 'Bouvet Island',
            'BR' => 'Brazil',
            'IO' => 'British Indian Ocean Territory',
            'BN' => 'Brunei Darussalam',
            'BG' => 'Bulgaria',
            'BF' => 'Burkina Faso',
            'BI' => 'Burundi',
            'KH' => 'Cambodia',
            'CM' => 'Cameroon',
            'CA' => 'Canada',
            'CV' => 'Cape Verde',
            'KY' => 'Cayman Islands',
            'CF' => 'Central African Republic',
            'TD' => 'Chad',
            'CL' => 'Chile',
            'CN' => 'China',
            'CX' => 'Christmas Island',
            'CC' => 'Cocos (Keeling) Islands',
            'CO' => 'Colombia',
            'KM' => 'Comoros',
            'CG' => 'Congo',
            'CD' => 'Congo, Democratic Republic',
            'CK' => 'Cook Islands',
            'CR' => 'Costa Rica',
            'CI' => 'Cote D\'Ivoire',
            'HR' => 'Croatia',
            'CU' => 'Cuba',
            'CY' => 'Cyprus',
            'CZ' => 'Czech Republic',
            'DK' => 'Denmark',
            'DJ' => 'Djibouti',
            'DM' => 'Dominica',
            'DO' => 'Dominican Republic',
            'EC' => 'Ecuador',
            'EG' => 'Egypt',
            'SV' => 'El Salvador',
            'GQ' => 'Equatorial Guinea',
            'ER' => 'Eritrea',
            'EE' => 'Estonia',
            'ET' => 'Ethiopia',
            'FK' => 'Falkland Islands (Malvinas)',
            'FO' => 'Faroe Islands',
            'FJ' => 'Fiji',
            'FI' => 'Finland',
            'FR' => 'France',
            'GF' => 'French Guiana',
            'PF' => 'French Polynesia',
            'TF' => 'French Southern Territories',
            'GA' => 'Gabon',
            'GM' => 'Gambia',
            'GE' => 'Georgia',
            'DE' => 'Germany',
            'GH' => 'Ghana',
            'GI' => 'Gibraltar',
            'GR' => 'Greece',
            'GL' => 'Greenland',
            'GD' => 'Grenada',
            'GP' => 'Guadeloupe',
            'GU' => 'Guam',
            'GT' => 'Guatemala',
            'GG' => 'Guernsey',
            'GN' => 'Guinea',
            'GW' => 'Guinea-Bissau',
            'GY' => 'Guyana',
            'HT' => 'Haiti',
            'HM' => 'Heard Island & Mcdonald Islands',
            'VA' => 'Holy See (Vatican City State)',
            'HN' => 'Honduras',
            'HK' => 'Hong Kong',
            'HU' => 'Hungary',
            'IS' => 'Iceland',
            'IN' => 'India',
            'ID' => 'Indonesia',
            'IR' => 'Iran, Islamic Republic Of',
            'IQ' => 'Iraq',
            'IE' => 'Ireland',
            'IM' => 'Isle Of Man',
            'IL' => 'Israel',
            'IT' => 'Italy',
            'JM' => 'Jamaica',
            'JP' => 'Japan',
            'JE' => 'Jersey',
            'JO' => 'Jordan',
            'KZ' => 'Kazakhstan',
            'KE' => 'Kenya',
            'KI' => 'Kiribati',
            'KR' => 'Korea',
            'KW' => 'Kuwait',
            'KG' => 'Kyrgyzstan',
            'LA' => 'Lao People\'s Democratic Republic',
            'LV' => 'Latvia',
            'LB' => 'Lebanon',
            'LS' => 'Lesotho',
            'LR' => 'Liberia',
            'LY' => 'Libyan Arab Jamahiriya',
            'LI' => 'Liechtenstein',
            'LT' => 'Lithuania',
            'LU' => 'Luxembourg',
            'MO' => 'Macao',
            'MK' => 'Macedonia',
            'MG' => 'Madagascar',
            'MW' => 'Malawi',
            'MY' => 'Malaysia',
            'MV' => 'Maldives',
            'ML' => 'Mali',
            'MT' => 'Malta',
            'MH' => 'Marshall Islands',
            'MQ' => 'Martinique',
            'MR' => 'Mauritania',
            'MU' => 'Mauritius',
            'YT' => 'Mayotte',
            'MX' => 'Mexico',
            'FM' => 'Micronesia, Federated States Of',
            'MD' => 'Moldova',
            'MC' => 'Monaco',
            'MN' => 'Mongolia',
            'ME' => 'Montenegro',
            'MS' => 'Montserrat',
            'MA' => 'Morocco',
            'MZ' => 'Mozambique',
            'MM' => 'Myanmar',
            'NA' => 'Namibia',
            'NR' => 'Nauru',
            'NP' => 'Nepal',
            'NL' => 'Netherlands',
            'AN' => 'Netherlands Antilles',
            'NC' => 'New Caledonia',
            'NZ' => 'New Zealand',
            'NI' => 'Nicaragua',
            'NE' => 'Niger',
            'NG' => 'Nigeria',
            'NU' => 'Niue',
            'NF' => 'Norfolk Island',
            'MP' => 'Northern Mariana Islands',
            'NO' => 'Norway',
            'OM' => 'Oman',
            'PK' => 'Pakistan',
            'PW' => 'Palau',
            'PS' => 'Palestinian Territory, Occupied',
            'PA' => 'Panama',
            'PG' => 'Papua New Guinea',
            'PY' => 'Paraguay',
            'PE' => 'Peru',
            'PH' => 'Philippines',
            'PN' => 'Pitcairn',
            'PL' => 'Poland',
            'PT' => 'Portugal',
            'PR' => 'Puerto Rico',
            'QA' => 'Qatar',
            'RE' => 'Reunion',
            'RO' => 'Romania',
            'RU' => 'Russian Federation',
            'RW' => 'Rwanda',
            'BL' => 'Saint Barthelemy',
            'SH' => 'Saint Helena',
            'KN' => 'Saint Kitts And Nevis',
            'LC' => 'Saint Lucia',
            'MF' => 'Saint Martin',
            'PM' => 'Saint Pierre And Miquelon',
            'VC' => 'Saint Vincent And Grenadines',
            'WS' => 'Samoa',
            'SM' => 'San Marino',
            'ST' => 'Sao Tome And Principe',
            'SA' => 'Saudi Arabia',
            'SN' => 'Senegal',
            'RS' => 'Serbia',
            'SC' => 'Seychelles',
            'SL' => 'Sierra Leone',
            'SG' => 'Singapore',
            'SK' => 'Slovakia',
            'SI' => 'Slovenia',
            'SB' => 'Solomon Islands',
            'SO' => 'Somalia',
            'ZA' => 'South Africa',
            'GS' => 'South Georgia And Sandwich Isl.',
            'ES' => 'Spain',
            'LK' => 'Sri Lanka',
            'SD' => 'Sudan',
            'SR' => 'Suriname',
            'SJ' => 'Svalbard And Jan Mayen',
            'SZ' => 'Swaziland',
            'SE' => 'Sweden',
            'CH' => 'Switzerland',
            'SY' => 'Syrian Arab Republic',
            'TW' => 'Taiwan',
            'TJ' => 'Tajikistan',
            'TZ' => 'Tanzania',
            'TH' => 'Thailand',
            'TL' => 'Timor-Leste',
            'TG' => 'Togo',
            'TK' => 'Tokelau',
            'TO' => 'Tonga',
            'TT' => 'Trinidad And Tobago',
            'TN' => 'Tunisia',
            'TR' => 'Turkey',
            'TM' => 'Turkmenistan',
            'TC' => 'Turks And Caicos Islands',
            'TV' => 'Tuvalu',
            'UG' => 'Uganda',
            'UA' => 'Ukraine',
            'AE' => 'United Arab Emirates',
            'GB' => 'United Kingdom',
            'US' => 'United States',
            'UM' => 'United States Outlying Islands',
            'UY' => 'Uruguay',
            'UZ' => 'Uzbekistan',
            'VU' => 'Vanuatu',
            'VE' => 'Venezuela',
            'VN' => 'Viet Nam',
            'VG' => 'Virgin Islands, British',
            'VI' => 'Virgin Islands, U.S.',
            'WF' => 'Wallis And Futuna',
            'EH' => 'Western Sahara',
            'YE' => 'Yemen',
            'ZM' => 'Zambia',
            'ZW' => 'Zimbabwe',
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_countries', $countries );

    }

    /**
     * Helper method to retrieve output types for Generate Nearby Cities / ZIP Codes
     *
     * @since   1.5.0
     *
     * @return  array      
     */
    public function get_nearby_cities_zipcodes_output_types() {

        // Define Output Types
        $output_types = array(
            'city'                          => __( 'City', 'page-generator-pro' ),
            'county'                        => __( 'County', 'page-generator-pro' ),
            'zipcode'                       => __( 'ZIP Code', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_nearby_cities_zipcodes_output_types', $output_types );

    }

    /**
     * Helper method to retrieve output types for Generate Nearby Cities / ZIP Codes
     *
     * @since   1.5.9
     *
     * @return  array      
     */
    public function get_phone_area_code_output_types() {

        // Define Output Types
        $output_types = array(
            'city'                          => __( 'City', 'page-generator-pro' ),
            'area_code'                     => __( 'Phone Area Code', 'page-generator-pro' ),
            'country_code'                  => __( 'Phone Country Code', 'page-generator-pro' ),
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_phone_area_output_types', $output_types );

    }

    /**
     * Helper method to retrieve Wikipedia languages
     *
     * @since   1.2.1
     *
     * @return  array    Languages
     */
    public function get_wikipedia_languages() {

        // Keys are Wikipedia subdomains e.g. ab.wikipedia.org
        // Values are the language names in English
        $languages = array(
            'ab' => 'Abkhazian',
            'ace' => 'Acehnese',
            'ady' => 'Adyghe',
            'aa' => 'Afar',
            'af' => 'Afrikaans',
            'ak' => 'Akan',
            'sq' => 'Albanian',
            'als' => 'Alemannic',
            'am' => 'Amharic',
            'ang' => 'Anglo-Saxon',
            'ar' => 'Arabic',
            'an' => 'Aragonese',
            'arc' => 'Aramaic',
            'hy' => 'Armenian',
            'roa-rup' => 'Aromanian',
            'as' => 'Assamese',
            'ast' => 'Asturian',
            'av' => 'Avar',
            'ay' => 'Aymara',
            'az' => 'Azerbaijani',
            'bm' => 'Bambara',
            'bjn' => 'Banjar',
            'map-bms' => 'Banyumasan',
            'ba' => 'Bashkir',
            'eu' => 'Basque',
            'bar' => 'Bavarian',
            'be' => 'Belarusian',
            'be-tarask' => 'Belarusian (Taraškievica)',
            'bn' => 'Bengali',
            'bh' => 'Bihari',
            'bpy' => 'Bishnupriya Manipuri',
            'bi' => 'Bislama',
            'bs' => 'Bosnian',
            'br' => 'Breton',
            'bug' => 'Buginese',
            'bg' => 'Bulgarian',
            'my' => 'Burmese',
            'bxr' => 'Buryat',
            'zh-yue' => 'Cantonese',
            'ca' => 'Catalan',
            'ceb' => 'Cebuano',
            'bcl' => 'Central Bicolano',
            'ch' => 'Chamorro',
            'cbk-zam' => 'Chavacano',
            'ce' => 'Chechen',
            'chr' => 'Cherokee',
            'chy' => 'Cheyenne',
            'ny' => 'Chichewa',
            'zh' => 'Chinese',
            'cho' => 'Choctaw',
            'cv' => 'Chuvash',
            'zh-classical' => 'Classical Chinese',
            'kw' => 'Cornish',
            'co' => 'Corsican',
            'cr' => 'Cree',
            'crh' => 'Crimean Tatar',
            'hr' => 'Croatian',
            'cs' => 'Czech',
            'da' => 'Danish',
            'dv' => 'Divehi',
            'nl' => 'Dutch',
            'nds-nl' => 'Dutch Low Saxon',
            'dz' => 'Dzongkha',
            'arz' => 'Egyptian Arabic',
            'eml' => 'Emilian-Romagnol',
            'en' => 'English',
            'myv' => 'Erzya',
            'eo' => 'Esperanto',
            'et' => 'Estonian',
            'ee' => 'Ewe',
            'ext' => 'Extremaduran',
            'fo' => 'Faroese',
            'hif' => 'Fiji Hindi',
            'fj' => 'Fijian',
            'fi' => 'Finnish',
            'frp' => 'Franco-Provençal',
            'fr' => 'French',
            'fur' => 'Friulian',
            'ff' => 'Fula',
            'gag' => 'Gagauz',
            'gl' => 'Galician',
            'gan' => 'Gan',
            'ka' => 'Georgian',
            'de' => 'German',
            'glk' => 'Gilaki',
            'gom' => 'Goan Konkani',
            'got' => 'Gothic',
            'el' => 'Greek',
            'kl' => 'Greenlandic',
            'gn' => 'Guarani',
            'gu' => 'Gujarati',
            'ht' => 'Haitian',
            'hak' => 'Hakka',
            'ha' => 'Hausa',
            'haw' => 'Hawaiian',
            'he' => 'Hebrew',
            'hz' => 'Herero',
            'mrj' => 'Hill Mari',
            'hi' => 'Hindi',
            'ho' => 'Hiri Motu',
            'hu' => 'Hungarian',
            'is' => 'Icelandic',
            'io' => 'Ido',
            'ig' => 'Igbo',
            'ilo' => 'Ilokano',
            'id' => 'Indonesian',
            'ia' => 'Interlingua',
            'ie' => 'Interlingue',
            'iu' => 'Inuktitut',
            'ik' => 'Inupiak',
            'ga' => 'Irish',
            'it' => 'Italian',
            'jam' => 'Jamaican Patois',
            'ja' => 'Japanese',
            'jv' => 'Javanese',
            'kbd' => 'Kabardian',
            'kab' => 'Kabyle',
            'xal' => 'Kalmyk',
            'kn' => 'Kannada',
            'kr' => 'Kanuri',
            'pam' => 'Kapampangan',
            'krc' => 'Karachay-Balkar',
            'kaa' => 'Karakalpak',
            'ks' => 'Kashmiri',
            'csb' => 'Kashubian',
            'kk' => 'Kazakh',
            'km' => 'Khmer',
            'ki' => 'Kikuyu',
            'rw' => 'Kinyarwanda',
            'ky' => 'Kirghiz',
            'rn' => 'Kirundi',
            'kv' => 'Komi',
            'koi' => 'Komi-Permyak',
            'kg' => 'Kongo',
            'ko' => 'Korean',
            'kj' => 'Kuanyama',
            'ku' => 'Kurdish (Kurmanji)',
            'ckb' => 'Kurdish (Sorani)',
            'lad' => 'Ladino',
            'lbe' => 'Lak',
            'lo' => 'Lao',
            'ltg' => 'Latgalian',
            'la' => 'Latin',
            'lv' => 'Latvian',
            'lez' => 'Lezgian',
            'lij' => 'Ligurian',
            'li' => 'Limburgish',
            'ln' => 'Lingala',
            'lt' => 'Lithuanian',
            'jbo' => 'Lojban',
            'lmo' => 'Lombard',
            'nds' => 'Low Saxon',
            'dsb' => 'Lower Sorbian',
            'lg' => 'Luganda',
            'lb' => 'Luxembourgish',
            'mk' => 'Macedonian',
            'mai' => 'Maithili',
            'mg' => 'Malagasy',
            'ms' => 'Malay',
            'ml' => 'Malayalam',
            'mt' => 'Maltese',
            'gv' => 'Manx',
            'mi' => 'Maori',
            'mr' => 'Marathi',
            'mh' => 'Marshallese',
            'mzn' => 'Mazandarani',
            'mhr' => 'Meadow Mari',
            'cdo' => 'Min Dong',
            'zh-min-nan' => 'Min Nan',
            'min' => 'Minangkabau',
            'xmf' => 'Mingrelian',
            'mwl' => 'Mirandese',
            'mdf' => 'Moksha',
            'mo' => 'Moldovan',
            'mn' => 'Mongolian',
            'mus' => 'Muscogee',
            'nah' => 'Nahuatl',
            'na' => 'Nauruan',
            'nv' => 'Navajo',
            'ng' => 'Ndonga',
            'nap' => 'Neapolitan',
            'ne' => 'Nepali',
            'new' => 'Newar',
            'pih' => 'Norfolk',
            'nrm' => 'Norman',
            'frr' => 'North Frisian',
            'lrc' => 'Northern Luri',
            'se' => 'Northern Sami',
            'nso' => 'Northern Sotho',
            'no' => 'Norwegian (Bokmål)',
            'nn' => 'Norwegian (Nynorsk)',
            'nov' => 'Novial',
            'ii' => 'Nuosu',
            'oc' => 'Occitan',
            'cu' => 'Old Church Slavonic',
            'or' => 'Oriya',
            'om' => 'Oromo',
            'os' => 'Ossetian',
            'pfl' => 'Palatinate German',
            'pi' => 'Pali',
            'pag' => 'Pangasinan',
            'pap' => 'Papiamentu',
            'ps' => 'Pashto',
            'pdc' => 'Pennsylvania German',
            'fa' => 'Persian',
            'pcd' => 'Picard',
            'pms' => 'Piedmontese',
            'pl' => 'Polish',
            'pnt' => 'Pontic',
            'pt' => 'Portuguese',
            'pa' => 'Punjabi',
            'qu' => 'Quechua',
            'ksh' => 'Ripuarian',
            'rmy' => 'Romani',
            'ro' => 'Romanian',
            'rm' => 'Romansh',
            'ru' => 'Russian',
            'rue' => 'Rusyn',
            'sah' => 'Sakha',
            'sm' => 'Samoan',
            'bat-smg' => 'Samogitian',
            'sg' => 'Sango',
            'sa' => 'Sanskrit',
            'sc' => 'Sardinian',
            'stq' => 'Saterland Frisian',
            'sco' => 'Scots',
            'gd' => 'Scottish Gaelic',
            'sr' => 'Serbian',
            'sh' => 'Serbo-Croatian',
            'st' => 'Sesotho',
            'sn' => 'Shona',
            'scn' => 'Sicilian',
            'szl' => 'Silesian',
            'simple' => 'Simple English',
            'sd' => 'Sindhi',
            'si' => 'Sinhalese',
            'sk' => 'Slovak',
            'sl' => 'Slovenian',
            'so' => 'Somali',
            'azb' => 'Southern Azerbaijani',
            'es' => 'Spanish',
            'srn' => 'Sranan',
            'su' => 'Sundanese',
            'sw' => 'Swahili',
            'ss' => 'Swati',
            'sv' => 'Swedish',
            'tl' => 'Tagalog',
            'ty' => 'Tahitian',
            'tg' => 'Tajik',
            'ta' => 'Tamil',
            'roa-tara' => 'Tarantino',
            'tt' => 'Tatar',
            'te' => 'Telugu',
            'tet' => 'Tetum',
            'th' => 'Thai',
            'bo' => 'Tibetan',
            'ti' => 'Tigrinya',
            'tpi' => 'Tok Pisin',
            'to' => 'Tongan',
            'ts' => 'Tsonga',
            'tn' => 'Tswana',
            'tum' => 'Tumbuka',
            'tr' => 'Turkish',
            'tk' => 'Turkmen',
            'tyv' => 'Tuvan',
            'tw' => 'Twi',
            'udm' => 'Udmurt',
            'uk' => 'Ukrainian',
            'hsb' => 'Upper Sorbian',
            'ur' => 'Urdu',
            'ug' => 'Uyghur',
            'uz' => 'Uzbek',
            've' => 'Venda',
            'vec' => 'Venetian',
            'vep' => 'Vepsian',
            'vi' => 'Vietnamese',
            'vo' => 'Volapük',
            'fiu-vro' => 'Võro',
            'wa' => 'Walloon',
            'war' => 'Waray',
            'cy' => 'Welsh',
            'vls' => 'West Flemish',
            'fy' => 'West Frisian',
            'pnb' => 'Western Punjabi',
            'wo' => 'Wolof',
            'wuu' => 'Wu',
            'xh' => 'Xhosa',
            'yi' => 'Yiddish',
            'yo' => 'Yoruba',
            'diq' => 'Zazaki',
            'zea' => 'Zeelandic',
            'za' => 'Zhuang',
            'zu' => 'Zulu',
        );

        // Return filtered results
        return apply_filters( 'page_generator_pro_common_get_wikipedia_languages', $languages );

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.3
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}