<?php
/**
 * WP-CLI Command: Generate Terms
 * 
 * @package Page Generator Pro
 * @author  Tim Carr
 * @version 1.6.1
 */
class Page_Generator_Pro_CLI_Generate_Terms {

    /**
     * Generates Terms based on the given group's settings
     *
     * @since   1.6.1
     *
     * @param   array   $args           Group ID (123) or Group IDs (123,456)
     * @param   array   $arguments      Array of associative arguments
     */
    public function __invoke( $args, $arguments ) {

        WP_CLI::log( 'Generate: Started' );
        WP_CLI::log( '--' );

        // Add the group ID(s) to the associative arguments
        if ( strpos( $args[0], ',' ) !== false ) {
            $arguments['group_id'] = explode( ',', $args[0] );
        } else {
            $arguments['group_id'] = absint( $args[0] );
        }

        // If the group_id argument is an array, we're generating multiple groups
        if ( is_array( $arguments['group_id'] ) ) {
            foreach ( $arguments['group_id'] as $group_id ) {
                $this->generate_terms( $group_id, $arguments );
            }
        } else {
            $this->generate_terms( $arguments['group_id'], $arguments );
        }

        WP_CLI::log( 'Generated: Finished' );
    
    }

    /**
     * Generates Terms based on the given group's settings
     *
     * @since   1.6.1
     *
     * @param   int     $group_id       Group ID
     * @param   array   $arguments      Array of associative arguments
     */
    private function generate_terms( $group_id, $arguments ) {

        // Get group
        $group = Page_Generator_Pro_Groups_Terms::get_instance()->get_settings( $group_id );
        if ( ! $group ) {
            WP_CLI::error( __( 'Group ID #' . $group_id . ': Could not get settings', 'page-generator-pro' ) );
        }

        // Replace the Group's settings with any arguments specified in the command
        if ( isset( $arguments['method'] ) ) {
            $group['method'] = $arguments['method'];
        }
        if ( isset( $arguments['overwrite'] ) && $arguments['overwrite'] == 1 ) {
            $group['overwrite'] = absint( $arguments['overwrite'] );
        }
        if ( isset( $arguments['number_of_terms'] ) && $arguments['number_of_terms'] > 0 ) {
            $group['numberOfPosts'] = absint( $arguments['number_of_terms'] );
        }
        if ( isset( $arguments['resume_index'] ) && $arguments['resume_index'] > 0 ) {
            $group['resumeIndex'] = absint( $arguments['resume_index'] );
        }

        // Calculate how many pages could be generated
        $number_of_pages_to_generate = Page_Generator_Pro_Generate::get_instance()->get_max_number_of_pages( $group );
        if ( is_wp_error( $number_of_pages_to_generate ) ) {
            WP_CLI::error( 'Group ID #' . $group_id . ': ' . $result->get_error_message() );
        }

        // If no limit specified, set one now
        if ( empty( $group['numberOfPosts'] ) ) {
            if ( $group['method'] == 'random' ) {
                $group['numberOfPosts'] = 10;
            } else {
                $group['numberOfPosts'] = $number_of_pages_to_generate;
            }
        }

        // If the requested Number of Posts exceeds the Number of Pages that could be generated,
        // set Number of Posts to match the Number of Pages that could be generated
        if ( $group['numberOfPosts'] > $number_of_pages_to_generate ) {
            $group['numberOfPosts'] = $number_of_pages_to_generate;
        }

        // Run a loop to generate each Term
        for ( $i = $group['resumeIndex']; $i < ( $group['numberOfPosts'] + $group['resumeIndex'] ); $i++ ) {
            // Run
            $start  = microtime( true );
            $result = Page_Generator_Pro_Generate::get_instance()->generate_term( $group_id, $i, false );
            $end    = microtime( true );

            // Bail if something went wrong
            if ( is_wp_error( $result ) ) {
                WP_CLI::error( 'Group ID #' . $group_id . ': ' . ( $i + 1 ) . '/' . ( $group['numberOfPosts'] + $group['resumeIndex'] ) . ': ' . $result->get_error_message() );
            }

            // Output success result and performance
            $data = array(
                'url'               => $result['url'],
                'keywords_terms'    => $result['keywords_terms'],
                'start'             => $start,
                'end'               => $end,
                'duration'          => round ( ( $end - $start ), 2 ),
                'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
                'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
            );

            // Build message and output
            $message = array(
                'Group ID #' . $group_id . ': ' . ( $i + 1 ) . '/' . ( $group['numberOfPosts'] + $group['resumeIndex'] ) . ': Created ' . $data['url'] . ' in ' . $data['duration'] . ' seconds.  Memory Usage / Peak: ' . $data['memory_usage'] . '/' . $data['memory_peak_usage'] . 'MB',
            );
            foreach ( $data['keywords_terms'] as $keyword => $term ) {
                $message[] = '{' . $keyword . '}: ' . $term;
            }
            $message[] = '--';
            WP_CLI::success( implode( "\n", $message ) );
        }

    } 

}

/**
 * WP-CLI Command: Terms Test
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.6.1
 */
class Page_Generator_Pro_CLI_Test_Terms {

    /**
     * Generates one Term based on the given group's settings
     *
     * @since   1.6.1
     *
     * @param   array   $args   Array of arguments (group ID, current index)
     */
    public function __invoke( $args ) {

        // Sanitize inputs
        $arguments['group_id'] = absint( $args[0] );

        // Run
        $start  = microtime( true );
        $result = Page_Generator_Pro_Generate::get_instance()->generate_term( $arguments['group_id'], 0, true );
        $end    = microtime( true );

        // Output success or error
        if ( is_wp_error( $result ) ) {
            WP_CLI::error( $result->get_error_message() );
        } else {
            // Output success result and performance
            $data = array(
                'url'               => $result['url'],
                'keywords_terms'    => $result['keywords_terms'],
                'start'             => $start,
                'end'               => $end,
                'duration'          => round ( ( $end - $start ), 2 ),
                'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
                'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
            );

            // Build message and output
            $message = array(
                'Group ID #' . $group_id . ': Created ' . $data['url'] . ' in ' . $data['duration'] . ' seconds.  Memory Usage / Peak: ' . $data['memory_usage'] . '/' . $data['memory_peak_usage'] . 'MB',
            );
            foreach ( $data['keywords_terms'] as $keyword => $term ) {
                $message[] = '{' . $keyword . '}: ' . $term;
            }
            WP_CLI::success( implode( "\n", $message ) );
        }

    } 

}

/**
 * WP-CLI Command: Delete Generated Terms
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.7.6
 */
class Page_Generator_Pro_CLI_Delete_Generated_Terms {

    /**
     * Deletes all generated content for the given Group ID
     *
     * @since   1.7.6
     *
     * @param   array   $args   Array of arguments (group ID)
     */
    public function __invoke( $args ) {

        // Sanitize inputs
        $arguments['group_id']       = absint( $args[0] );

        // Run
        $start  = microtime( true );
        $result = Page_Generator_Pro_Generate::get_instance()->delete_terms( $arguments['group_id'] );
        $end    = microtime( true );

        // Output success or error
        if ( is_wp_error( $result ) ) {
            WP_CLI::error( $result->get_error_message() );
        } else {
            $data = array(
                'start'             => $start,
                'end'               => $end,
                'duration'          => round ( ( $end - $start ), 2 ),
                'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
                'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
            );

            // Build message and output
            $message = array(
                'Group ID #' . $arguments['group_id'] . ': Deleted Generated Terms in ' . $data['duration'] . ' seconds.  Memory Usage / Peak: ' . $data['memory_usage'] . '/' . $data['memory_peak_usage'] . 'MB',
            );
            WP_CLI::success( implode( "\n", $message ) );
        }

    } 

}

/**
 * WP-CLI Command: Generate Content
 * 
 * @package Page Generator Pro
 * @author  Tim Carr
 * @version 1.2.1
 */
class Page_Generator_Pro_CLI_Generate_Content {

    /**
     * Generates Pages, Posts or CPTs based on the given group's settings
     *
     * @since   1.2.1
     *
     * @param   array   $args           Group ID (123) or Group IDs (123,456)
     * @param   array   $arguments      Array of associative arguments
     */
    public function __invoke( $args, $arguments ) {

        WP_CLI::log( 'Generate: Started' );
        WP_CLI::log( '--' );

        // Add the group ID(s) to the associative arguments
        if ( strpos( $args[0], ',' ) !== false ) {
            $arguments['group_id'] = explode( ',', $args[0] );
        } else {
            $arguments['group_id'] = absint( $args[0] );
        }

        // If the group_id argument is an array, we're generating multiple groups
        if ( is_array( $arguments['group_id'] ) ) {
            foreach ( $arguments['group_id'] as $group_id ) {
                $this->generate( $group_id, $arguments );
            }
        } else {
            $this->generate( $arguments['group_id'], $arguments );
        }

        WP_CLI::log( 'Generated: Finished' );
    
    }

    /**
     * Generates Pages, Posts or CPTs based on the given group's settings
     *
     * @since   1.5.3
     *
     * @param   int     $group_id       Group ID
     * @param   array   $arguments      Array of associative arguments
     */
    private function generate( $group_id, $arguments ) {

        // Get group
        $group = Page_Generator_Pro_Groups::get_instance()->get_settings( $group_id );
        if ( ! $group ) {
            WP_CLI::error( __( 'Group ID #' . $group_id . ': Could not get settings', 'page-generator-pro' ) );
        }

        // Replace the Group's settings with any arguments specified in the command
        if ( isset( $arguments['method'] ) ) {
            $group['method'] = $arguments['method'];
        }
        if ( isset( $arguments['overwrite'] ) && $arguments['overwrite'] == 1 ) {
            $group['overwrite'] = absint( $arguments['overwrite'] );
        }
        if ( isset( $arguments['number_of_posts'] ) && $arguments['number_of_posts'] > 0 ) {
            $group['numberOfPosts'] = absint( $arguments['number_of_posts'] );
        }
        if ( isset( $arguments['resume_index'] ) && $arguments['resume_index'] > 0 ) {
            $group['resumeIndex'] = absint( $arguments['resume_index'] );
        }

        // Calculate how many pages could be generated
        $number_of_pages_to_generate = Page_Generator_Pro_Generate::get_instance()->get_max_number_of_pages( $group );
        if ( is_wp_error( $number_of_pages_to_generate ) ) {
            WP_CLI::error( 'Group ID #' . $group_id . ': ' . $result->get_error_message() );
        }

        // If no limit specified, set one now
        if ( empty( $group['numberOfPosts'] ) ) {
            if ( $group['method'] == 'random' ) {
                $group['numberOfPosts'] = 10;
            } else {
                $group['numberOfPosts'] = $number_of_pages_to_generate;
            }
        }

        // If the requested Number of Posts exceeds the Number of Pages that could be generated,
        // set Number of Posts to match the Number of Pages that could be generated
        if ( $group['numberOfPosts'] > $number_of_pages_to_generate ) {
            $group['numberOfPosts'] = $number_of_pages_to_generate;
        }

        // Run a loop to generate each page
        for ( $i = $group['resumeIndex']; $i < ( $group['numberOfPosts'] + $group['resumeIndex'] ); $i++ ) {
            // Run
            $start  = microtime( true );
            $result = Page_Generator_Pro_Generate::get_instance()->generate( $group_id, $i, false );
            $end    = microtime( true );

            // Bail if something went wrong
            if ( is_wp_error( $result ) ) {
                WP_CLI::error( 'Group ID #' . $group_id . ': ' . ( $i + 1 ) . '/' . ( $group['numberOfPosts'] + $group['resumeIndex'] ) . ': ' . $result->get_error_message() );
            }

            // Output success result and performance
            $data = array(
                'url'               => $result['url'],
                'keywords_terms'    => $result['keywords_terms'],
                'start'             => $start,
                'end'               => $end,
                'duration'          => round ( ( $end - $start ), 2 ),
                'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
                'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
            );
        
            // Build message and output
            $message = array(
                'Group ID #' . $group_id . ': ' . ( $i + 1 ) . '/' . ( $group['numberOfPosts'] + $group['resumeIndex'] ) . ': Created ' . $data['url'] . ' in ' . $data['duration'] . ' seconds.  Memory Usage / Peak: ' . $data['memory_usage'] . '/' . $data['memory_peak_usage'] . 'MB',
            );
            foreach ( $data['keywords_terms'] as $keyword => $term ) {
                $message[] = '{' . $keyword . '}: ' . $term;
            }
            $message[] = '--';
            WP_CLI::success( implode( "\n", $message ) );
        }

    } 

}

/**
 * WP-CLI Command: Test Content
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.2.1
 */
class Page_Generator_Pro_CLI_Test_Content {

    /**
     * Generates one Page in Draft mode based on the given group's settings
     *
     * @since   1.2.1
     *
     * @param   array   $args   Array of arguments (group ID, current index)
     */
    public function __invoke( $args ) {

        // Sanitize inputs
        $arguments['group_id']       = absint( $args[0] );

        // Run
        $start  = microtime( true );
        $result = Page_Generator_Pro_Generate::get_instance()->generate_content( $arguments['group_id'], 0, true );
        $end    = microtime( true );

        // Output success or error
        if ( is_wp_error( $result ) ) {
            WP_CLI::error( $result->get_error_message() );
        } else {
            $data = array(
                'url'               => $result['url'],
                'keywords_terms'    => $result['keywords_terms'],
                'start'             => $start,
                'end'               => $end,
                'duration'          => round ( ( $end - $start ), 2 ),
                'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
                'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
            );

            // Build message and output
            $message = array(
                'Group ID #' . $group_id . ': Created ' . $data['url'] . ' in ' . $data['duration'] . ' seconds.  Memory Usage / Peak: ' . $data['memory_usage'] . '/' . $data['memory_peak_usage'] . 'MB',
            );
            foreach ( $data['keywords_terms'] as $keyword => $term ) {
                $message[] = '{' . $keyword . '}: ' . $term;
            }
            WP_CLI::success( implode( "\n", $message ) );
        }

    } 

}

/**
 * WP-CLI Command: Delete Generated Content
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.7.6
 */
class Page_Generator_Pro_CLI_Delete_Generated_Content {

    /**
     * Deletes all generated content for the given Group ID
     *
     * @since   1.7.6
     *
     * @param   array   $args   Array of arguments (group ID)
     */
    public function __invoke( $args ) {

        // Sanitize inputs
        $arguments['group_id']       = absint( $args[0] );

        // Run
        $start  = microtime( true );
        $result = Page_Generator_Pro_Generate::get_instance()->delete_content( $arguments['group_id'] );
        $end    = microtime( true );

        // Output success or error
        if ( is_wp_error( $result ) ) {
            WP_CLI::error( $result->get_error_message() );
        } else {
            $data = array(
                'start'             => $start,
                'end'               => $end,
                'duration'          => round ( ( $end - $start ), 2 ),
                'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
                'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
            );

            // Build message and output
            $message = array(
                'Group ID #' . $arguments['group_id'] . ': Deleted Generated Content in ' . $data['duration'] . ' seconds.  Memory Usage / Peak: ' . $data['memory_usage'] . '/' . $data['memory_peak_usage'] . 'MB',
            );
            WP_CLI::success( implode( "\n", $message ) );
        }

    } 

}

/**
 * WP-CLI Command: Groups List
 * 
 * @package  Page_Generator_Pro
 * @author   Tim Carr
 * @version  1.2.1
 */
class Page_Generator_Pro_CLI_Groups_List {

    /**
     * Lists all Page Generator Pro Groups in table format within the CLI
     *
     * @since   1.5.3
     */
    public function __invoke() {

        // Get all Groups
        $groups = Page_Generator_Pro_Groups::get_instance()->get_all();

        // Build array for WP-CLI Table
        $groups_table = array();
        foreach ( $groups as $group_id => $group ) {
            $groups_table[ $index ] = array(
                'ID'                        => $group_id,
                'title'                     => $group['title'],
                'generated_pages_count'     => $group['generated_pages_count'],
            );
        }

        // Output
        \WP_CLI\Utils\format_items( 'table', $groups_table, array(
            'ID', 'title', 'generated_pages_count',
        ) );

    } 

}

// Register WP-CLI commands here

// Generate Content
WP_CLI::add_command( 'page-generator-pro-generate', 'Page_Generator_Pro_CLI_Generate_Content', array(
    'shortdesc' => __( 'Generates Pages / Posts / Custom Post Types for the given Generate Group ID.', 'page-generator-pro' ),
    'synopsis'  => array(
        array(
            'type'     => 'positional',
            'name'     => 'group_id',
            'optional' => false,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'method',
            'optional' => true,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'overwrite',
            'optional' => true,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'number_of_posts',
            'optional' => true,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'resume_index',
            'optional' => true,
            'multiple' => false,
        ),
    ),
    'when' => 'before_wp_load',
) );

// Test Content
WP_CLI::add_command( 'page-generator-pro-test', 'Page_Generator_Pro_CLI_Test_Content', array(
    'shortdesc' => __( 'Generates one Page / Post / CPT for the given Generate Group ID, storing it as a Draft. Use this to test your settings.', 'page-generator-pro' ),
    'synopsis'  => array(
        array(
            'type'     => 'positional',
            'name'     => 'group_id',
            'optional' => false,
            'multiple' => false,
        ),
    ),
    'when' => 'before_wp_load',
) );

// Delete Content
WP_CLI::add_command( 'page-generator-pro-delete-generated-content', 'Page_Generator_Pro_CLI_Delete_Generated_Content', array(
    'shortdesc' => __( 'Deletes all generated content for the given Group ID.', 'page-generator-pro' ),
    'synopsis'  => array(
        array(
            'type'     => 'positional',
            'name'     => 'group_id',
            'optional' => false,
            'multiple' => false,
        ),
    ),
    'when' => 'before_wp_load',
) );

// Generate Terms
WP_CLI::add_command( 'page-generator-pro-generate-terms', 'Page_Generator_Pro_CLI_Generate_Terms', array(
    'shortdesc' => __( 'Generates Terms for the given Generate Group ID.', 'page-generator-pro' ),
    'synopsis'  => array(
        array(
            'type'     => 'positional',
            'name'     => 'group_id',
            'optional' => false,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'method',
            'optional' => true,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'overwrite',
            'optional' => true,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'number_of_terms',
            'optional' => true,
            'multiple' => false,
        ),
        array(
            'type'     => 'assoc',
            'name'     => 'resume_index',
            'optional' => true,
            'multiple' => false,
        ),
    ),
    'when' => 'before_wp_load',
) );

// Test Terms
WP_CLI::add_command( 'page-generator-pro-test-terms', 'Page_Generator_Pro_CLI_Test_Terms', array(
    'shortdesc' => __( 'Generates one Term for the given Generate Group ID. Use this to test your settings.', 'page-generator-pro' ),
    'synopsis'  => array(
        array(
            'type'     => 'positional',
            'name'     => 'group_id',
            'optional' => false,
            'multiple' => false,
        ),
    ),
    'when' => 'before_wp_load',
) );

// Delete Terms
WP_CLI::add_command( 'page-generator-pro-delete-generated-terms', 'Page_Generator_Pro_CLI_Delete_Generated_Terms', array(
    'shortdesc' => __( 'Deletes all generated terms for the given Group ID', 'page-generator-pro' ),
    'synopsis'  => array(
        array(
            'type'     => 'positional',
            'name'     => 'group_id',
            'optional' => false,
            'multiple' => false,
        ),
    ),
    'when' => 'before_wp_load',
) );

// List Groups
WP_CLI::add_command( 'page-generator-pro-groups-list', 'Page_Generator_Pro_CLI_Groups_List', array(
    'shortdesc' => __( 'Lists all Groups in the CLI.', 'page-generator-pro' ),
    'when'      => 'before_wp_load',
) );