<?php
/**
 * AJAX class
 * 
 * @package Page Generator Pro
 * @author  Tim Carr
 * @version 1.0.0
 */
class Page_Generator_Pro_AJAX {

    /**
     * Holds the class object.
     *
     * @since   1.1.3
     *
     * @var     object
     */
    public static $instance;

    /**
     * Constructor
     *
     * @since   1.0.0
     */
    public function __construct() {

        // TinyMCE
        add_action( 'wp_ajax_page_generator_pro_output_tinymce_google_maps_modal', array( $this, 'output_tinymce_google_maps_modal' ) );
        add_action( 'wp_ajax_page_generator_pro_output_tinymce_related_links_modal', array( $this, 'output_tinymce_related_links_modal' ) );
        add_action( 'wp_ajax_page_generator_pro_output_tinymce_wikipedia_modal', array( $this, 'output_tinymce_wikipedia_modal' ) );
        add_action( 'wp_ajax_page_generator_pro_output_tinymce_yelp_modal', array( $this, 'output_tinymce_yelp_modal' ) );
        add_action( 'wp_ajax_page_generator_pro_output_tinymce_youtube_modal', array( $this, 'output_tinymce_youtube_modal' ) );

        // Generate
        add_action( 'wp_ajax_page_generator_pro_generate', array( $this, 'generate' ) ); // Backward compat < 1.6.0
        add_action( 'wp_ajax_page_generator_pro_generate_content', array( $this, 'generate_content' ) );
        add_action( 'wp_ajax_page_generator_pro_generate_term', array( $this, 'generate_term' ) );
        
    }
    
    /**
     * Loads the view for the Google Maps TinyMCE modal.
     *
     * @since   1.5.2
     */
    public function output_tinymce_google_maps_modal() {

        // Load View
        require_once( Page_Generator_Pro::get_instance()->plugin->folder . '/views/admin/tinymce-google-maps.php' ); 
        die();

    }

    /**
     * Loads the view for the Related Links modal.
     *
     * @since   1.5.9
     */
    public function output_tinymce_related_links_modal() {

        // Load View
        require_once( Page_Generator_Pro::get_instance()->plugin->folder . '/views/admin/tinymce-related-links.php' ); 
        die();

    }

    /**
     * Loads the view for the Wikipedia TinyMCE modal.
     *
     * @since   1.5.2
     */
    public function output_tinymce_wikipedia_modal() {

        // Load View
        require_once( Page_Generator_Pro::get_instance()->plugin->folder . '/views/admin/tinymce-wikipedia.php' ); 
        die();

    }

    /**
     * Loads the view for the Yelp TinyMCE modal.
     *
     * @since   1.5.2
     */
    public function output_tinymce_yelp_modal() {

        // Load View
        require_once( Page_Generator_Pro::get_instance()->plugin->folder . '/views/admin/tinymce-yelp.php' ); 
        die();

    }

    /**
     * Loads the view for the YouTube TinyMCE modal.
     *
     * @since   1.5.2
     */
    public function output_tinymce_youtube_modal() {

        // Load View
        require_once( Page_Generator_Pro::get_instance()->plugin->folder . '/views/admin/tinymce-youtube.php' ); 
        die();

    }

    /**
     * Backward compat function for Page Generator Pro 1.6.0 and older.
     *
     * Passes the request to generate_content()
     *
     * @since   1.0.0
     */
    public function generate() {

        $this->generate_content();

    }

    /**
     * Generates a Page, Post or CPT
     *
     * @since   1.6.1
     */
    public function generate_content() {

        // Validate
        $result = $this->generate_validation();

        // Run
        $start  = microtime( true );
        $result = Page_Generator_Pro_Generate::get_instance()->generate_content( $result['group_id'], $result['current_index'], false );
        $end    = microtime( true );

        // Return
        $this->generate_return( $result, $start, $end );

    } 

    /**
     * Generates a Term
     *
     * @since   1.6.1
     */
    public function generate_term() {

        // Validate
        $result = $this->generate_validation();

        // Run
        $start  = microtime( true );
        $result = Page_Generator_Pro_Generate::get_instance()->generate_term( $result['group_id'], $result['current_index'], false );
        $end    = microtime( true );

        // Return
        $this->generate_return( $result, $start, $end );

    }

    /**
     * Runs validation when AJAX calls are made to generate content or terms,
     * returning the Group ID and Current Index.
     *
     * @since   1.6.1
     *
     * @return  array   Group ID and Current Index
     */
    private function generate_validation() {

        // Sanitize inputs
        if ( ! isset( $_POST['id'] ) ) {
            wp_send_json_error( __( 'No group ID was specified!', 'page-generator-pro' ) );
            die();
        }
        if ( ! isset( $_POST['current_index'] ) ) {
            wp_send_json_error( __( 'No current index was specified!', 'page-generator-pro' ) );
            die();
        }
        
        return array(
            'group_id'      => absint( $_POST['id'] ),
            'current_index' => absint( $_POST['current_index'] ),
        );

    }

    /**
     * Returns the generation result as a JSON error or success
     *
     * @since   1.6.1
     *
     * @param   mixed   $result     WP_Error | string URL
     * @param   float   $start      Start Execution Time
     * @param   float   $end        End Execution Time
     */
    private function generate_return( $result, $start, $end ) {

        // Return error or success JSON
        if ( is_wp_error( $result ) ) {
            wp_send_json_error( $result->get_error_code() . ': ' . $result->get_error_message() );
        }

        // If here, run routine worked
        wp_send_json_success( array(
            'url'               => $result['url'],
            'keywords_terms'    => $result['keywords_terms'],
            'start'             => $start,
            'end'               => $end,
            'duration'          => round ( ( $end - $start ), 2 ),
            'memory_usage'      => round( memory_get_usage() / 1024 / 1024 ),
            'memory_peak_usage' => round( memory_get_peak_usage() / 1024 / 1024 ),
        ) );

    }

    /**
     * Returns the singleton instance of the class.
     *
     * @since   1.1.3
     *
     * @return  object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}

// Load the class
$page_generator_pro_ajax = Page_Generator_Pro_AJAX::get_instance();