<?php
/**
 * Administration class
 * 
 * @package Page Generator Pro
 * @author  Tim Carr
 * @version 1.0.0
 */
class Page_Generator_Pro_Admin {

    /**
     * Holds the class object.
     *
     * @since   1.1.3
     *
     * @var     object
     */
    public static $instance;

    /**
     * Holds the base object.
     *
     * @since   1.2.1
     *
     * @var     object
     */
    public $base;

    /**
     * Holds success and error notices
     *
     * @since   1.3.8
     *
     * @var     array
     */
    public $notices = array(
        'success'   => array(),
        'error'     => array(),
    );

    /**
     * Constructor
     * 
     * @since   1.0.0
     */
    public function __construct() {

        // Admin CSS, JS and Menu
        add_filter( 'wpzinc_admin_body_class', array( $this, 'admin_body_class' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'admin_scripts_css' ) );
        add_action( 'admin_menu', array( $this, 'admin_menu' ), 8 );
        add_action( 'parent_file', array( $this, 'admin_menu_hierarchy_correction' ), 999 );

        // Settings Panels
        add_filter( 'page_generator_pro_setting_panel', array( $this, 'register_settings_panel' ), 1 );
        add_action( 'page_generator_pro_setting_panel-page-generator-pro-google', array( $this, 'settings_screen_google' ), 1 );
        add_action( 'page_generator_pro_setting_panel-page-generator-pro-geonames', array( $this, 'settings_screen_geonames' ), 1 );

        // Import & Export
        add_action( 'page-generator-pro-import', array( $this, 'import' ) );
        add_filter( 'page-generator-pro-export', array( $this, 'export' ) );

        // Localization
        add_action( 'plugins_loaded', array( $this, 'load_language_files' ) );

    }

    /**
     * Registers screen names that should add the wpzinc class to the <body> tag
     *
     * @since   1.6.1
     *
     * @param   array   $screens    Screen Names
     * @return  array               Screen Names
     */
    public function admin_body_class( $screens ) {

        // Add Post Types
        $screens[] = Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name;

        // Return
        return $screens;

    }

    /**
     * Enqueues CSS and JS
     *
     * @since   1.0.0
     */
    public function admin_scripts_css() {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Get current screen
        $screen = get_current_screen();
        
        // CSS - always load
        add_editor_style( $this->base->plugin->url . 'assets/css/admin.css' );
        wp_enqueue_style( $this->base->plugin->name . '-admin', $this->base->plugin->url . 'assets/css/admin.css', array(), $this->base->plugin->version );

        // Don't load anything else if we're not on a Plugin screen
        if ( strpos( $screen->base, $this->base->plugin->name ) === false && $screen->post_type != Page_Generator_Pro_PostType::get_instance()->post_type_name && $screen->taxonomy != Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name ) {
            return;
        }

        // Prevents errors with meta boxes and Yoast
        wp_enqueue_media();
    
        // Plugin Admin
        // These scripts are registered in _modules/dashboard/dashboard.php
        // JS
        wp_enqueue_script( 'wpzinc-admin-conditional' );
        wp_enqueue_script( 'wpzinc-admin-inline-search' );
        wp_enqueue_script( 'wpzinc-admin-selectize' );
        wp_enqueue_script( 'wpzinc-admin-tags' );
        wp_enqueue_script( 'wpzinc-admin' );

        // CSS
        wp_enqueue_style( 'wpzinc-admin-selectize' );
        
        // Plugin JS
        wp_enqueue_script( 'jquery-ui-progressbar' );
        wp_enqueue_script( 'jquery-ui-sortable' );
        wp_enqueue_script( $this->base->plugin->name . '-synchronous-ajax', $this->base->plugin->url . 'assets/js/synchronous-ajax.js', array( 'jquery' ), $this->base->plugin->version, true );
        wp_enqueue_script( $this->base->plugin->name . '-admin', $this->base->plugin->url . 'assets/js/admin.js', array( 'jquery' ), $this->base->plugin->version, true );

        // CSS
        if ( class_exists( 'Page_Generator' ) ) {
            // Hide 'Add New' if a Group exists
            $number_of_groups = Page_Generator_Pro_Groups::get_instance()->get_count();
            if ( $number_of_groups > 0 ) {
                ?>
                <style type="text/css">body.post-type-page-generator-pro a.page-title-action { display: none; }</style>
                <?php
            }
        }
        
    }

    /**
     * Add the Plugin to the WordPress Administration Menu
     *
     * @since   1.0.0
     */
    public function admin_menu() {

        global $submenu;
        
        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Licensing
        add_menu_page( $this->base->plugin->displayName, $this->base->plugin->displayName, 'manage_options', $this->base->plugin->name, array( $this, 'licensing_screen' ), 'dashicons-format-aside' );
        add_submenu_page( $this->base->plugin->name, __( 'Licensing', $this->base->plugin->name ), __( 'Licensing', $this->base->plugin->name ), 'manage_options', $this->base->plugin->name, array( $this, 'licensing_screen' ) );
        
        // Bail if the product is not licensed
        if ( ! $this->base->licensing->check_license_key_valid() ) {
            return;
        }

        // Licensed - add additional menu entries
        $settings_page      = add_submenu_page( $this->base->plugin->name, __( 'Settings', $this->base->plugin->name ), __( 'Settings', $this->base->plugin->name ), 'manage_options', $this->base->plugin->name . '-settings', array( $this, 'settings_screen' ) );    
        $keywords_page      = add_submenu_page( $this->base->plugin->name, __( 'Keywords', $this->base->plugin->name ), __( 'Keywords', $this->base->plugin->name ), 'manage_options', $this->base->plugin->name . '-keywords', array( $this, 'keywords_screen' ) );    
        $groups_page        = add_submenu_page( $this->base->plugin->name, __( 'Generate Content', $this->base->plugin->name ), __( 'Generate Content', $this->base->plugin->name ), 'manage_options', 'edit.php?post_type=' . Page_Generator_Pro_PostType::get_instance()->post_type_name );    
        $groups_tax_page    = add_submenu_page( $this->base->plugin->name, __( 'Generate Terms', $this->base->plugin->name ), __( 'Generate Terms', $this->base->plugin->name ), 'manage_options', 'edit-tags.php?taxonomy=' . Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name );    
        $generate_page      = add_submenu_page( $this->base->plugin->name, __( 'Generate', $this->base->plugin->name ), __( 'Generate', $this->base->plugin->name ), 'manage_options', $this->base->plugin->name . '-generate', array( $this, 'generate_screen' ) );    

        // Add any other submenu pages now
        do_action( 'page_generator_pro_admin_menu' );

    }

    /**
     * Ensures this Plugin's top level Admin menu remains open when the user clicks on:
     * - Generate Content
     * - Generate Terms
     *
     * This prevents the 'wrong' admin menu being open (e.g. Posts)
     *
     * @since   1.2.3
     *
     * @param   string  $parent_file    Parent Admin Menu File Name
     * @return  string                  Parent Admin Menu File Name
     */
    public function admin_menu_hierarchy_correction( $parent_file ) {

        global $current_screen;

        // If we're creating or editing a Content Group, set the $parent_file to this Plugin's registered menu name
        if ( $current_screen->base == 'post' && $current_screen->post_type == Page_Generator_Pro_PostType::get_instance()->post_type_name ) {
            // The free version uses a different top level filename
            if ( class_exists( 'Page_Generator' ) ) {
                return $this->base->plugin->name . '-keywords';
            }

            return $this->base->plugin->name;
        }

        // If we're creating or editing a Term Group, set the $parent_file to this Plugin's registered menu name
        if ( ( $current_screen->base == 'edit-tags' || $current_screen->base == 'term' ) && $current_screen->taxonomy == Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name ) {
            return $this->base->plugin->name;
        }

        return $parent_file;

    }

    /**
     * Registers Settings Panel(s) for the WordPress Administration Settings screen
     *
     * @since   1.0.0
     *
     * @param   array $panels   Settings Panels (key/value pairs)
     * @return  array           Panels
     */
    public function register_settings_panel( $panels ) {
        
        $panels[ $this->base->plugin->name . '-google' ] = array(
            'label' => __( 'Google', $this->base->plugin->name ),
            'icon'  => 'dashicons-admin-generic',
        );
        $panels[ $this->base->plugin->name . '-geonames' ] = array(
            'label' => __( 'GeoNames', $this->base->plugin->name ),
            'icon'  => 'dashicons-admin-generic',
        );
        
        return $panels;

    }

    /**
     * Outputs the Licensing Screen
     *
     * @since   1.0.0
     */
    public function licensing_screen() {

        include_once( $this->base->plugin->folder . '_modules/licensing/views/licensing.php' ); 

    }

    /**
     * Output the Settings Screen
     * Save POSTed data from the Administration Panel into a WordPress option
     *
     * @since 1.0
     */
    public function settings_screen() {

        // Get Page
        $page = ( isset( $_REQUEST['page'] ) ? sanitize_text_field( $_REQUEST['page'] ) : '' );

        // Get registered settings panels and active tab
        $panels = apply_filters( 'page_generator_pro_setting_panel', array() );
        $tab   = ( isset( $_GET['tab'] ) ? $_GET['tab'] : $this->base->plugin->name . '-google' );

        // Maybe save settings
        $result = $this->save_settings( $tab );
        if ( is_string( $result ) ) {
            // Error - add to array of errors for output
            $this->notices['error'][] = $result;
        } elseif ( $result === true ) {
            // Success
            $this->notices['success'][] = __( 'Settings saved successfully.', $this->base->plugin->name ); 
        }

        // Load View
        include_once( $this->base->plugin->folder . '/views/admin/settings.php' ); 

    }

    /**
     * Outputs the Google Settings Screen within Page Generator Pro > Settings
     *
     * @since   1.2.1
     */
    public function settings_screen_google() {

        // Load view
        include_once( $this->base->plugin->folder . '/views/admin/settings-google.php' );    
    
    }

    /**
     * Outputs the Geonames Settings Screen within Page Generator Pro > Settings
     *
     * @since   1.5.2
     */
    public function settings_screen_geonames() {

        // Load view
        include_once( $this->base->plugin->folder . '/views/admin/settings-geonames.php' );    
    
    }

    /**
     * Outputs the Keywords Screens
     *
     * @since 1.0.0
     */
    public function keywords_screen() {

        // Get Page
        $page = ( isset( $_REQUEST['page'] ) ? sanitize_text_field( $_REQUEST['page'] ) : '' );

        // Get command
        $cmd = ( ( isset($_GET['cmd'] ) ) ? $_GET['cmd'] : '' );
        switch ( $cmd ) {

            /**
             * Generate Nearby Cities
             */
            case 'form-city':
                // Generate nearby cities
                $result = $this->generate_nearby_cities();
                if ( is_wp_error( $result ) ) {
                    $this->notices['error'][] = $result->get_error_message();
                } elseif ( is_numeric( $result ) ) {
                    // Success
                    $this->notices['success'][] = __( 'Nearby City / ZIP Code Keyword generated successfully.', $this->base->plugin->name ); 
                }

                // Get countries and output types
                $countries      = Page_Generator_Pro_Common::get_instance()->get_countries();
                $output_types   = Page_Generator_Pro_Common::get_instance()->get_nearby_cities_zipcodes_output_types();

                // View
                $view = 'views/admin/keywords-form-city.php';

                break;

            /**
             * Generate Phone Area Codes
             */
            case 'form-phone':
                // Generate phone area codes
                $result = $this->generate_phone_area_codes();
                if ( is_wp_error( $result ) ) {
                    $this->notices['error'][] = $result->get_error_message();
                } elseif ( is_numeric( $result ) ) {
                    // Success
                    $this->notices['success'][] = __( 'Phone Area Codes generated successfully.', $this->base->plugin->name ); 
                }

                // Get countries and output types
                $countries      = Page_Generator_Pro_Geo::get_instance()->get_phone_area_code_countries();
                $output_types   = Page_Generator_Pro_Common::get_instance()->get_phone_area_code_output_types();

                // View
                $view = 'views/admin/keywords-form-phone.php';

                break;

            /**
             * Import
             */
            case 'form-import-csv':
                $result = $this->import_csv();
                if ( is_wp_error( $result ) ) {
                    $this->notices['error'][] = $result->get_error_message();
                } elseif ( is_numeric( $result ) ) {
                    // Success
                    $this->notices['success'][] = sprintf( __( '%s Keywords imported successfully.', $this->base->plugin->name ), $result ); 
                }

                // View
                $view = 'views/admin/keywords-form-import-csv.php';
                break;

            /**
             * Add / Edit Keyword
             */
            case 'form':
                // Get keyword from POST data or DB
                if ( isset( $_POST['keyword'] ) ) {
                    // Get keyword from POST data
                    $keyword = stripslashes_deep( $_POST );
                } else if ( isset( $_GET['id'] ) ) {
                    // Get keyword from DB
                    $keyword = Page_Generator_Pro_Keywords::get_instance()->get_by_id( $_GET['id'] );
                }

                // Save keyword
                $result = $this->save_keyword();
                if ( is_wp_error( $result ) ) {
                    $this->notices['error'][] = $result->get_error_message();
                } else if ( is_numeric( $result ) ) {
                    $this->notices['success'][] = __( 'Keyword saved successfully.', $this->base->plugin->name ); 

                    // Fetch saved keyword from DB
                    $keyword = Page_Generator_Pro_Keywords::get_instance()->get_by_id( absint( $result ) );
                }

                // View
                $view = 'views/admin/keywords-form.php';
                
                break;

            /**
            * Index Table
            */
            case 'delete':
            default: 
                // Delete keywords
                $result = $this->delete_keywords();
                if ( is_string( $result ) ) {
                    // Error - add to array of errors for output
                    $this->notices['error'][] = $result;
                } elseif ( $result === true ) {
                    // Success
                    $this->notices['success'][] = __( 'Keyword(s) deleted successfully.', $this->base->plugin->name ); 
                }
                
                // View
                $view = 'views/admin/keywords-table.php';
                
                break;   

        }

        // Load View
        include_once( $this->base->plugin->folder . $view ); 

    }

    /**
     * Save Settings Screen
     *
     * @since 1.0.0
     *
     * @param string $type Plugin Name / Type
     * @return mixed Error String on error, true on success
     */
    public function save_settings( $type ) {

        // For oAuth authorization, the values we need to save (or destroy) are in the URL
        if ( isset( $_REQUEST['wpzinc_provider'] ) ) {
            // Save oauth data
            return Page_Generator_Pro_Settings::get_instance()->update_settings( $this->base->plugin->name . '-' . $_REQUEST['wpzinc_provider'], array(
                'oauth_token'       => $_REQUEST['wpzinc_oauth_token'],
                'oauth_token_secret'=> $_REQUEST['wpzinc_oauth_token_secret'],
            ) );
        }
        if ( isset( $_REQUEST['wpzinc_provider_disconnect'] ) ) {
            // Delete oAuth data
            return Page_Generator_Pro_Settings::get_instance()->delete_settings( $this->base->plugin->name . '-' . $_REQUEST['wpzinc_provider_disconnect'] );
        }

        // Check if a POST request was made
        if ( ! isset( $_POST['submit'] ) ) {
            return false;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return __( 'Nonce field is missing. Settings NOT saved.', $this->base->plugin->name );
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], $this->base->plugin->name ) ) {
            return __( 'Invalid nonce specified. Settings NOT saved.', $this->base->plugin->name );
        }

        // Save and return
        return Page_Generator_Pro_Settings::get_instance()->update_settings( $type, $_POST[ $type ] );

    }

    /**
     * Generate Nearby Cities
     *
     * @since   1.0.0
     *
     * @return  mixed   WP_Error | int
     */
    public function generate_nearby_cities() {

        // Check if a POST request was made
        if ( ! isset( $_POST['submit'] ) ) {
            return false;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return new WP_Error( __( 'Nonce field is missing. Settings NOT saved.', $this->base->plugin->name ) );
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], 'generate_nearby_cities' ) ) {
            return new WP_Error( __( 'Invalid nonce specified. Settings NOT saved.', $this->base->plugin->name ) );
        }

        // Setup instances
        $geo = Page_Generator_Pro_Geo::get_instance();

        // If a Google Geocoding API key has been specified, use it instead of the class default.
        $google_geocoding_api_key = Page_Generator_Pro_Settings::get_instance()->get_setting( $this->base->plugin->name . '-google', 'google_geocoding_api_key' );
        if ( ! empty( $google_geocoding_api_key ) ) {
            $geo->api_key = $google_geocoding_api_key;
        }

        // Get latitude and longitude of our start point; bail if an error occured
        $lat_lng = $geo->google_get_lat_lng( sanitize_text_field( $_POST['city'] ), sanitize_text_field( $_POST['country'] ) );
        if ( is_wp_error( $lat_lng ) ) {
            return $lat_lng;
        }

        // Fetch data at zip code level (includes cities + counties) or city level,
        // depending on the output type array
        foreach ( $_POST['output_type'] as $output_type ) {
            if ( $output_type == 'zipcode' ) {
                // If a Geonames username has been specified, use it instead of the class default.
                $username = Page_Generator_Pro_Settings::get_instance()->get_setting( $this->base->plugin->name . '-geonames', 'username' );

                // Get Zip Codes
                $terms = $geo->get_nearby_zipcodes( $lat_lng['lat'], $lat_lng['lng'], absint( $_POST['radius'] ), $username );
                
                // We don't need to iterate through other output types, as we have the most precise data needed
                break;
            }

            // City / county lookups
            $terms = $geo->get_nearby_cities( $lat_lng['lat'], $lat_lng['lng'], absint( $_POST['radius'] ) );
        }

        // Bail if an error occured
        if ( is_wp_error( $terms ) ) {
            return $terms;
        }

        // Generate keyword
        $keyword = array();
        $keyword['keyword'] = sanitize_text_field( $_POST['keyword'] );
        $keyword['data']    = '';

        // If we're including the original value, do so now
        // This might be a City or a ZIP Code
        if ( isset( $_POST['include_city'] ) ) {
            $keyword['data'] .= sanitize_text_field( $_POST['city'] ) . "\n"; 
        }

        // Build the keyword data based on the output type formatting
        $formatted_terms = array();
        foreach ( $terms as $i => $term ) {
            // Define array to build output order for this term
            $formatted_terms[ $i ] = array();

            // Build array
            foreach ( $_POST['output_type'] as $output_type ) {
                $formatted_terms[ $i ][] = ( isset( $term[ $output_type ] ) ? $term[ $output_type ] : ( $output_type == 'city' ? $term : '' ) );
            }

            // Remove any empty array values, and implode into a string
            $formatted_terms[ $i ] = implode( ', ', array_filter( $formatted_terms[ $i ] ) );
        }

        // Remove duplicates
        // This should never occur, but it's a good fallback just in case
        $formatted_terms = array_values( array_unique( $formatted_terms ) );

        // Add Terms to keyword data
        $keyword['data'] .= implode( "\n", $formatted_terms ); 

        // Save keyword
        return Page_Generator_Pro_Keywords::get_instance()->save( $keyword );

    }

    /**
     * Generate Phone Area Codes
     *
     * @since   1.5.9
     *
     * @return  mixed   WP_Error | int
     */
    public function generate_phone_area_codes() {

        // Check if a POST request was made
        if ( ! isset( $_POST['submit'] ) ) {
            return false;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return new WP_Error( __( 'Nonce field is missing. Phone Area Codes NOT generated.', $this->base->plugin->name ) );
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], 'generate_phone_area_codes' ) ) {
            return new WP_Error( __( 'Invalid nonce specified. Phone Area Codes NOT generated.', $this->base->plugin->name ) );
        }

        // Fetch all phone area codes for the given country
        $terms = Page_Generator_Pro_Geo::get_instance()->get_phone_area_codes( sanitize_text_field( $_POST['country'] ) );

        // Bail if no area codes were found
        if ( ! $terms ) {
            return new WP_Error( __( 'No phone area codes could be found for the selected country. Please choose a different country.', $this->base->plugin->name ) );
        }

        // Generate keyword
        $keyword = array();
        $keyword['keyword'] = sanitize_text_field( $_POST['keyword'] );
        $keyword['data']    = '';

        // Build the keyword data based on the output type formatting
        $formatted_terms = array();
        foreach ( $terms as $i => $term ) {
            // Define array to build output order for this term
            $formatted_terms[ $i ] = array();

            // Build array
            foreach ( $_POST['output_type'] as $output_type ) {
                $formatted_terms[ $i ][] = ( isset( $term[ $output_type ] ) ? $term[ $output_type ] : '' );
            }

            // Remove any empty array values, and implode into a string
            $formatted_terms[ $i ] = implode( ', ', array_filter( $formatted_terms[ $i ] ) );
        }

        // Remove duplicates
        // This should never occur, but it's a good fallback just in case
        $formatted_terms = array_values( array_unique( $formatted_terms ) );

        // Add Terms to keyword data
        $keyword['data'] .= implode( "\n", $formatted_terms ); 

        // Save keyword
        return Page_Generator_Pro_Keywords::get_instance()->save( $keyword );

    }

    /**
     * Imports the given CSV file into multiple keywords, each with terms
     *
     * @since   1.7.3
     *
     * @return  mixed   WP_Error | int
     */
    public function import_csv() {

        // Check if a POST request was made
        if ( ! isset( $_POST['submit'] ) ) {
            return false;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return new WP_Error( __( 'Nonce field is missing. CSV file NOT imported.', $this->base->plugin->name ) );
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], 'import_csv' ) ) {
            return new WP_Error( __( 'Invalid nonce specified. CSV file NOT imported.', $this->base->plugin->name ) );
        }

        // Get keywords instance
        $instance = Page_Generator_Pro_Keywords::get_instance();

        // Get form data
        $keywords_location = sanitize_text_field( $_POST['keywords_location'] );

        // Fetch keywords and terms from CSV file
        return $instance->import_csv_file_data( $keywords_location );

    }

    /**
     * Save Keyword
     *
     * @since   1.0.0
     *
     * @return  mixed   WP_Error | int
     */
    public function save_keyword() {

        // Check if a POST request was made
        if ( ! isset( $_POST['submit'] ) ) {
            return false;
        }

        // Run security checks
        // Missing nonce 
        if ( ! isset( $_POST[ $this->base->plugin->name . '_nonce' ] ) ) { 
            return new WP_Error( __( 'Nonce field is missing. Settings NOT saved.', $this->base->plugin->name ) );
        }

        // Invalid nonce
        if ( ! wp_verify_nonce( $_POST[ $this->base->plugin->name . '_nonce' ], 'save_keyword' ) ) {
            return new WP_Error( __( 'Invalid nonce specified. Settings NOT saved.', $this->base->plugin->name ) );
        }

        // Get ID
        $id = ( ( isset($_REQUEST['id'] ) && ! empty( $_REQUEST['id'] ) ) ? $_REQUEST['id'] : '' );

        // Get keywords instance
        $instance = Page_Generator_Pro_Keywords::get_instance();

        // Build keyword and import data
        $keyword = array();
        $keyword['keyword'] = sanitize_text_field( $_POST['keyword'] );
        
        // If there is no POSTed data, generate terms automatically
        if ( ! isset( $_POST['data'] ) || empty( $_POST['data'] ) ) {
            // Setup Webanalyzer API and run query
            $api = new Page_Generator_Pro_Webanalyzer_API();
            $terms = $api->synonyms( $keyword['keyword'] );

            // Add terms to keyword if they exist in the response
            if ( ! is_wp_error( $terms ) && isset( $terms->success ) && isset( $terms->data ) && is_array( $terms->data ) && count( $terms->data ) > 0 ) {
                $keyword['data'] = implode( "\n", $terms->data );
            }
        } else {
            $keyword['data'] = $instance->import_text_file_data( $_POST['data'] );
        }

        // Save Keyword
        return $instance->save( $keyword, $id );

    }

    /**
     * Delete Keyword(s), if commands to do this have been sent in the request
     *
     * @since   1.0.0
     *
     * @return  mixed   WP_Error | true
     */
    public function delete_keywords() {

        // Check an action exists
        if ( ! isset( $_POST['action'] ) && ! isset( $_POST['action2'] ) && ! isset( $_GET['cmd'] ) ) {
            return false;
        }

        // Set flag
        $deleted = false;

        // Get instance
        $instance = Page_Generator_Pro_Keywords::get_instance();

        // Bulk Delete
        if ( ( isset( $_POST['action2'] ) && $_POST['action2'] != '-1' ) ||
             ( isset( $_POST['action'] ) && $_POST['action'] != '-1' ) ) {

            $result = $instance->delete( $_POST['ids'] );
            if ( is_wp_error( $result ) ) {
                return $result;
            }

            $deleted = true;

        }

        // Single Delete
        if ( isset( $_GET['cmd'] ) && $_GET['cmd'] == 'delete' ) {
            $result = $instance->delete( $_GET['id'] );
            if ( is_wp_error( $result ) ) {
                return $result;
            }

            $deleted = true;

        }

        // If a delete happened, return true
        return $deleted;

    }

    /**
     * Generates content for the given Group and Group Type
     *
     * @since   1.2.3
     */
    public function generate_screen() {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Check a Group ID and Type was supplied
        if ( ! isset( $_REQUEST['id'] ) ) {
            return;
        }
        $id = absint( $_REQUEST['id'] );
        $type = ( isset( $_REQUEST['type'] ) ? sanitize_text_field( $_REQUEST['type'] ) : 'content' );

        // Get group settings and return to Group URL, depending on the type
        switch ( $type ) {
            case 'term':
                $settings = Page_Generator_Pro_Groups_Terms::get_instance()->get_settings( $id );
                $return_url = admin_url( 'term.php?taxonomy=' . Page_Generator_Pro_Taxonomy::get_instance()->taxonomy_name . '&tag_ID=' . $id );
                break;

            case 'content':
            default:
                $settings = Page_Generator_Pro_Groups::get_instance()->get_settings( $id );
                $return_url = admin_url( 'post.php?post=' . $id . '&amp;action=edit' );
                break;
        }
        
        // Bail if no settings
        if ( ! $settings ) {
            return;
        }

        // Calculate how many pages could be generated
        $number_of_pages_to_generate = Page_Generator_Pro_Generate::get_instance()->get_max_number_of_pages( $settings );
          
        // Check that the number of posts doesn't exceed the maximum that can be generated
        if ( $settings['numberOfPosts'] > $number_of_pages_to_generate ) {
            $settings['numberOfPosts'] = $number_of_pages_to_generate;
        }  

        // If no limit specified, set one now
        if ( $settings['numberOfPosts'] == 0 ) {
            if ( $settings['method'] == 'random' ) {
                $settings['numberOfPosts'] = 10;
            } else {
                $settings['numberOfPosts'] = $number_of_pages_to_generate;
            }
        }

        // Code which executes the process is in the view, as we use sync JS
        $view = 'views/admin/generate-run.php';

        // Load View
        include_once( $this->base->plugin->folder . $view );

    }

    /**
     * Import data when Dashboard submodule import routine runs
     *
     * @since   1.0.7
     *
     * @param   array   $data   Data
     */
    public function import( $data ) {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Get instance
        $keywords_instance  = Page_Generator_Pro_Keywords::get_instance();
        $groups_instance    = Page_Generator_Pro_Groups::get_instance();
        $settings_instance  = Page_Generator_Pro_Settings::get_instance();

        // Keywords
        if ( isset( $data['keywords'] ) ) {
            foreach ( $data['keywords'] as $keyword ) {
                // Create keyword
                $keywords_instance->save( $keyword ); 
            }
        }

        // Groups
        if ( isset( $data['groups'] ) && is_array( $data['groups'] ) ) {
            // Determine whether the Groups data is from 1.2.2 or earlier (where we didn't use a CPT for Groups),
            // or is 1.2.3+ (where we use a CPT for Groups)
            foreach ( $data['groups'] as $group ) {
                // If a groupID key exists, this is from 1.2.2 or older
                if ( isset( $group['groupID'] ) ) {
                    // Import is from <= 1.2.2
                    $args = array(
                        'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
                        'post_status'   => 'publish',
                        'post_title'    => $group['settings']['title'],
                        'post_content'  => $group['settings']['content'],
                    );
                    $settings = $group['settings'];
                } else {
                    // Import is from >= 1.2.3
                    $args = array(
                        'post_type'     => Page_Generator_Pro_PostType::get_instance()->post_type_name,
                        'post_status'   => 'publish',
                        'post_title'    => $group['title'],
                        'post_content'  => $group['content'],
                    );
                    $settings = $group;
                }

                // Create group
                $id = wp_insert_post( $args );

                // Skip if something went wrong
                if ( is_wp_error( $id ) ) {
                    continue;
                }

                // Save group settings
                $groups_instance->save( $settings, $id ); 
            }
        }
        
        // Settings: Google
        if ( isset( $data['google'] ) ) {
            $settings_instance->update_settings( $this->base->plugin->name . '-google', $data['google'] );
        }

        // Settings: Geonames
        if ( isset( $data['geonames'] ) ) {
            $settings_instance->update_settings( $this->base->plugin->name . '-geonames', $data['geonames'] );
        }
        
    }
    
    /**
     * Export data when Dashboard submodule export routine runs
     *
     * @since   1.0.7
     *
     * @param   array   $data   Data
     * @return  array           Data
     */
    public function export( $data ) {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        // Keywords
        $data['keywords'] = Page_Generator_Pro_Keywords::get_instance()->get_all( 'id', 'ASC', -1 );

        // Groups
        $data['groups'] = Page_Generator_Pro_Groups::get_instance()->get_all();

        // Settings
        $data['google'] = Page_Generator_Pro_Settings::get_instance()->get_settings( $this->base->plugin->name . '-google' );
        $data['geonames'] = Page_Generator_Pro_Settings::get_instance()->get_settings( $this->base->plugin->name . '-geonames' );

        return $data;
        
    }

    /**
     * Loads plugin textdomain
     *
     * @since 1.0.0
     */
    public function load_language_files() {

        // Get base instance
        $this->base = ( class_exists( 'Page_Generator_Pro' ) ? Page_Generator_Pro::get_instance() : Page_Generator::get_instance() );

        load_plugin_textdomain( $this->base->plugin->name, false, $this->base->plugin->name . '/languages/' );

    } 

    /**
     * Returns the singleton instance of the class.
     *
     * @since 1.1.3
     *
     * @return object Class.
     */
    public static function get_instance() {

        if ( ! isset( self::$instance ) && ! ( self::$instance instanceof self ) ) {
            self::$instance = new self;
        }

        return self::$instance;

    }

}

// Load the class
$page_generator_pro_admin = Page_Generator_Pro_Admin::get_instance();